package org.gluu.oxd.licenser.server.service;

import com.google.common.base.Strings;
import com.google.common.io.BaseEncoding;
import com.google.inject.Inject;
import net.nicholaswilliams.java.licensing.encryption.RSAKeyPairGenerator;
import org.apache.commons.lang.RandomStringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gluu.oxd.license.client.js.LdapLicenseCrypt;
import org.gluu.oxd.license.client.lib.LicenseSerializationUtilities;
import org.gluu.oxd.license.client.js.Configuration;
import org.gluu.oxd.licenser.server.ldap.LdapStructure;
import org.gluu.persist.PersistenceEntryManager;
import org.gluu.search.filter.Filter;

import java.security.KeyPair;
import java.util.Collections;
import java.util.List;
import java.util.UUID;

/**
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 03/10/2014
 */

public class LicenseCryptService {

    private static final Logger LOG = LoggerFactory.getLogger(LicenseCryptService.class);

    @Inject
    PersistenceEntryManager ldapEntryManager;
    @Inject
    Configuration conf;
    @Inject
    LdapStructure ldapStructure;

    public LdapLicenseCrypt generate() {
        return generate("generated-"+ System.currentTimeMillis());
    }

    public LdapLicenseCrypt generate(String name) {
        RSAKeyPairGenerator generator = new RSAKeyPairGenerator();
        KeyPair keyPair = generator.generateKeyPair();

        final String privatePassword = randomPassword();
        final String publicPassword = randomPassword();
        final byte[] privateKeyBytes = LicenseSerializationUtilities.writeEncryptedPrivateKey(keyPair.getPrivate(), privatePassword.toCharArray());
        final byte[] publicKeyBytes = LicenseSerializationUtilities.writeEncryptedPublicKey(keyPair.getPublic(), publicPassword.toCharArray());
        return new LdapLicenseCrypt().
                setName(name).
                setPrivateKey(BaseEncoding.base64().encode(privateKeyBytes)).
                setPublicKey(BaseEncoding.base64().encode(publicKeyBytes)).
                setPrivatePassword(privatePassword).
                setPublicPassword(publicPassword).
                setLicensePassword(randomPassword());
    }

    public String randomPassword() {
        return RandomStringUtils.randomAlphanumeric(20);
    }

    public List<LdapLicenseCrypt> getAll() {
        try {
            Filter filter = Filter.createPresenceFilter("uniqueIdentifier");
            return ldapEntryManager.findEntries(ldapStructure.getLicenseCryptBaseDn(), LdapLicenseCrypt.class, filter);
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
        }
        return Collections.emptyList();
    }

    public LdapLicenseCrypt get(String dn) {
        return ldapEntryManager.find(LdapLicenseCrypt.class, dn);
    }

    public LdapLicenseCrypt getById(String id) {
        return get(dn(id));
    }

    public void merge(LdapLicenseCrypt entity) {
        try {
            ldapEntryManager.merge(entity);
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
        }
    }

    public void save(LdapLicenseCrypt entity) {
        try {
            setDnIfEmpty(entity);
            ldapEntryManager.persist(entity);
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
        }
    }

    private void setDnIfEmpty(LdapLicenseCrypt entity) {
        if (Strings.isNullOrEmpty(entity.getDn())) {
            String id = Strings.isNullOrEmpty(entity.getId()) ? UUID.randomUUID().toString() : entity.getId();
            entity.setId(id);
            entity.setDn(dn(id));
        }
    }

    public String dn(String id) {
        return String.format("uniqueIdentifier=%s,%s", id, ldapStructure.getLicenseCryptBaseDn());
    }

    public void remove(LdapLicenseCrypt entity) {
        try {
            ldapEntryManager.remove(entity);
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
        }
    }
}
