package org.gluu.oxd.licenser.server.service;

import com.google.inject.Inject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gluu.oxd.license.client.Jackson;
import org.gluu.oxd.license.client.js.LdapLicenseId;
import org.gluu.oxd.license.client.js.StatisticUpdateRequest;
import org.gluu.oxd.license.client.js.event.LdapClientEvent;

import javax.ws.rs.WebApplicationException;
import java.util.Date;

/**
 * @author Yuriy Zabrovarnyy
 */

public class StatisticUpdator {

    private static final Logger LOG = LoggerFactory.getLogger(StatisticUpdator.class);

    @Inject
    ValidationService validationService;
    @Inject
    StatisticService statisticService;

    public void update(StatisticUpdateRequest request) {
        if (request.getType() == StatisticUpdateRequest.Type.CLIENT) {
            updateClient(request);
            return;
        }

        LOG.error("Unknown statistic update type: " + request.getType());
        throw new WebApplicationException(ErrorService.response("Unknown statistic update type: " + request.getType()));
    }

    private void updateClient(StatisticUpdateRequest request) {
        final LdapLicenseId licenseId = validationService.getLicenseId(request.getProperties().get(StatisticUpdateRequest.LICENSE_ID));
        final String macAddress = validationService.validateNotEmpty(request.getProperties(), StatisticUpdateRequest.MAC_ADDRESS);
        final String clientId = validationService.validateNotEmpty(request.getProperties(), StatisticUpdateRequest.CLIENT_ID);
        final String oxdId = validationService.validateNotEmpty(request.getProperties(), StatisticUpdateRequest.OXD_ID);
        final String clientName = request.getProperties().get(StatisticUpdateRequest.CLIENT_NAME);
        final String isClientLocal = request.getProperties().get(StatisticUpdateRequest.IS_CLIENT_LOCAL);

        LdapClientEvent event = new LdapClientEvent();
        event.setMacAddress(macAddress);
        event.setClientId(clientId);
        event.setClientName(clientName);
        event.setOxdId(oxdId);
        event.setClientLocal(Boolean.parseBoolean(isClientLocal));
        event.setAppMetadata(Jackson.asJsonSilently(request.getAppMetadata()));
        event.setCreationDate(new Date());

        statisticService.saveClientEvent(event, licenseId.getLicenseId());

    }

}
