package org.gluu.oxd.licenser.server.guice;

import com.google.common.base.Preconditions;
import com.google.inject.AbstractModule;
import com.google.inject.Provides;
import com.google.inject.Singleton;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gluu.oxd.license.client.js.Configuration;
import org.gluu.oxd.license.client.js.JsonFileConfiguration;
import org.gluu.oxd.licenser.server.KeyPairService;
import org.gluu.oxd.licenser.server.LicenseGenerator;
import org.gluu.oxd.licenser.server.conf.ConfigurationFactory;
import org.gluu.oxd.licenser.server.ldap.Conf;
import org.gluu.oxd.licenser.server.service.LdapStructureChecker;
import org.gluu.oxd.licenser.server.service.LicenseCryptService;
import org.gluu.oxd.licenser.server.service.LicenseGeneratedStatService;
import org.gluu.oxd.licenser.server.service.LicenseIdService;
import org.gluu.oxd.licenser.server.service.StatisticService;
import org.gluu.oxd.licenser.server.service.StatisticUpdator;
import org.gluu.oxd.licenser.server.service.ValidationService;
import org.gluu.oxd.licenser.server.ws.GenerateLicenseWS;
import org.gluu.oxd.licenser.server.ws.MetadataWS;
import org.gluu.oxd.licenser.server.ws.StatisticWS;
import org.gluu.oxd.licenser.server.ws.UmaProtector;
import org.gluu.persist.PersistenceEntryManager;
import org.gluu.persist.PersistenceEntryManagerFactory;
import org.gluu.persist.exception.EntryPersistenceException;
import org.gluu.persist.ldap.impl.LdapEntryManagerFactory;
import org.gluu.persist.model.base.Entry;
import org.gluu.persist.model.base.GluuDummyEntry;
import org.gluu.util.Util;
import org.gluu.util.properties.FileConfiguration;
import org.gluu.util.security.PropertiesDecrypter;
import org.gluu.util.security.StringEncrypter;
import org.gluu.util.security.StringEncrypter.EncryptionException;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

/**
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 07/09/2014
 */

public class AppModule extends AbstractModule {

    private static final Logger LOG = LoggerFactory.getLogger(AppModule.class);

    private static final String ENCRYPTION_KEY = "123456789012345678901234567890";

    @Override
    protected void configure() {
        bind(KeyPairService.class);
        bind(LicenseGenerator.class);
        bind(ValidationService.class);
        bind(LicenseIdService.class);
        bind(LicenseCryptService.class);
        bind(LicenseGeneratedStatService.class);
        bind(StatisticService.class);
        bind(StatisticUpdator.class);

        bind(UmaProtector.class);

        // ws
        bind(GenerateLicenseWS.class);
        bind(MetadataWS.class);
        bind(StatisticWS.class);
    }

    @Provides
    @Singleton
    public PersistenceEntryManager provideLdapManager() throws StringEncrypter.EncryptionException {
		Properties connectionProperties = preparePersistanceProperties();

		PersistenceEntryManager persistenceEntryManager = getPersistenceEntryManagerFactory()
				.createEntryManager(connectionProperties);
		LOG.info("Created: {} with operation service: {}",
				new Object[] { persistenceEntryManager, persistenceEntryManager.getOperationService() });

		Entry base = persistenceEntryManager.find(GluuDummyEntry.class, "o=gluu");
        Preconditions.checkNotNull(base);

		return persistenceEntryManager;

    }

	protected Properties preparePersistanceProperties() throws EncryptionException {
    	final FileConfiguration fileConfiguration = ConfigurationFactory.getLdapConfiguration();
        final Properties decryptedConnectionProperties = PropertiesDecrypter.decryptProperties(StringEncrypter.instance(ENCRYPTION_KEY), fileConfiguration.getProperties());

        return decryptedConnectionProperties;
	}

    public PersistenceEntryManagerFactory getPersistenceEntryManagerFactory() {
    	LdapEntryManagerFactory persistenceEntryManagerFactory = new LdapEntryManagerFactory();

        return persistenceEntryManagerFactory;
    }

    @Provides
    @Singleton
    public JsonFileConfiguration provideJsonConfiguration() {
        InputStream stream = null;
        try {
            LOG.info("Configuration file location: {}", getConfigFileLocation());
            final File configFile = new File(getConfigFileLocation());
            if (configFile.exists()) {
                stream = new FileInputStream(configFile);
            } else {
                LOG.error("No configuration file. Fail to start! Location: " + getConfigFileLocation());
            }
            return Util.createJsonMapper().readValue(stream, JsonFileConfiguration.class);
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            return null;
        } finally {
            IOUtils.closeQuietly(stream);
        }
    }

    public String getConfigFileLocation() {
        return ConfigurationFactory.CONFIG_FILE_LOCATION;
    }

    public String getUmaProtectFileLocation() {
        return ConfigurationFactory.UMA_PROTECT_FILE_LOCATION;
    }

    @Provides
    @Singleton
    public Configuration provideConfiguration(PersistenceEntryManager ldapManager, JsonFileConfiguration jsonFileConfiguration) throws IOException {
        final String dn = ConfigurationFactory.getLdapConfiguration().getString("configurationEntryDN");
        try {
            final Conf conf = ldapManager.find(Conf.class, dn);
            if (conf != null) {
                final Configuration configuration = Util.createJsonMapper().readValue(conf.getConf(), Configuration.class);
                new LdapStructureChecker(ldapManager, configuration).checkLdapStructure();
                return configuration;
            } else {
                LOG.error("Conf object is null.");
            }
        } catch (EntryPersistenceException e) {
            LOG.trace(e.getMessage(), e);
            LOG.info("Unable to find configuration in LDAP, try to create configuration entry in LDAP... ");
            if (ConfigurationFactory.getLdapConfiguration().getBoolean("createLdapConfigurationEntryIfNotExist")) {
                if (jsonFileConfiguration != null) {
                    final Conf c = new Conf();
                    c.setDn(ConfigurationFactory.getLdapConfiguration().getString("configurationEntryDN"));
                    c.setConf(Util.createJsonMapper().writeValueAsString(jsonFileConfiguration));
                    LOG.info("Persisting ldap conf: " + c);
                    try {
                        ldapManager.persist(c);
                        LOG.info("Configuration entry is created in LDAP.");
                    } catch (Exception ex) {
                        LOG.error(e.getMessage(), ex);
                    }
                    new LdapStructureChecker(ldapManager, jsonFileConfiguration).checkLdapStructure();
                    return jsonFileConfiguration;
                } else {
                    LOG.error("Unable to read configuration from file: " + ConfigurationFactory.CONFIG_FILE_NAME);
                }
            } else {
                LOG.error("Skip configuration creation because createLdapConfigurationEntryIfNotExist is set to false instead of true in " + ConfigurationFactory.CONFIG_FILE_NAME);
            }
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
        }

        LOG.error("Failed to create configuration.");
        return null;
    }
}