package org.gluu.oxd.licenser.server.ws;

import com.google.inject.Inject;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.jboss.resteasy.annotations.interception.ServerInterceptor;
import org.jboss.resteasy.core.ResourceMethodInvoker;
import org.jboss.resteasy.core.ServerResponse;
import org.jboss.resteasy.spi.Failure;
import org.jboss.resteasy.spi.HttpRequest;
import org.jboss.resteasy.spi.interception.PreProcessInterceptor;
import org.gluu.oxd.license.client.Jackson;
import org.gluu.oxd.license.client.js.Configuration;
import org.gluu.oxd.licenser.server.conf.ConfigurationFactory;
import org.gluu.oxd.rs.protect.RsProtector;
import org.gluu.oxd.rs.protect.RsResource;
import org.gluu.oxd.rs.protect.StaticStorage;
import org.gluu.oxd.rs.protect.resteasy.*;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.ext.Provider;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.Collection;
import java.util.Collections;

/**
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 24/06/2016
 */
@Provider
@ServerInterceptor
public class UmaProtector implements PreProcessInterceptor {

    private static final Logger LOG = Logger.getLogger(UmaProtector.class);
    private static volatile boolean initialized = false;
    private static volatile boolean failedToInitialize = false;

    private static final String UGLY_HARDCODE = "ugly_hardcode_gerald_please_fix_rpt";

    private RptPreProcessInterceptor interceptor;
    private Configuration conf;

    @Inject
    public UmaProtector(Configuration conf) {
        try {
            this.conf = conf;
            LOG.info("Configuration : " + conf);

            initIfNeeded();
            interceptor = new RptPreProcessInterceptor(StaticStorage.get(ResourceRegistrar.class)) {
                @Override
                public String getPath(HttpRequest request) {
                    return PathPatcher.patchPath(super.getPath(request));
                }
            };
            LOG.info("UMA Protector started successfully.");
        } catch (Exception e) {
            LOG.error("Failed to initialize UMA Protector. " + e.getMessage(), e);
            failedToInitialize = true;
        }
    }

    private void initIfNeeded() {
        if (initialized) {
            return;
        }

        init();

        initialized = true;
    }

    private void init() {
        try {
            Collection<RsResource> values = resources();

            if (values.isEmpty()) {
                throw new RuntimeException("Failed to load UMA protection_document.");
            }

            LOG.info("Protection configuration: " + Jackson.asJsonSilently(values));

            ServiceProvider serviceProvider = new ServiceProvider(conf.getOpHost(), true);

            org.gluu.oxd.rs.protect.resteasy.Configuration umaLibConf = new org.gluu.oxd.rs.protect.resteasy.Configuration();
            umaLibConf.setOpHost(conf.getOpHost());
            umaLibConf.setTrustAll(true);
            umaLibConf.setUmaPatClientId(conf.getUmaPatClientId());
            umaLibConf.setUmaPatClientSecret(conf.getUmaPatClientSecret());

            ObtainPatProvider patProvider = new ObtainPatProvider(serviceProvider, umaLibConf);
            ResourceRegistrar resourceRegistrar = new ResourceRegistrar(patProvider, serviceProvider);

            resourceRegistrar.register(values);
            LOG.info("Resources are registered at AS: " + umaLibConf);

            StaticStorage.put(PatProvider.class, patProvider);
            StaticStorage.put(ResourceRegistrar.class, resourceRegistrar);

            LOG.info("Resource Server started successfully.");
        } catch (Exception e) {
            LOG.error("Failed to initialize UMA Protector. " + e.getMessage(), e);
            failedToInitialize = true;
        }
    }

    private Collection<RsResource> resources() {
        InputStream stream = null;
        try {
            LOG.info("UMA protect file location: " + ConfigurationFactory.UMA_PROTECT_FILE_LOCATION);
            final File configFile = new File(ConfigurationFactory.UMA_PROTECT_FILE_LOCATION);
            if (configFile.exists()) {
                stream = new FileInputStream(configFile);
            } else {
                LOG.error("No configuration file. Fail to start! Location: " + ConfigurationFactory.UMA_PROTECT_FILE_LOCATION);
            }
            return RsProtector.instance(stream).getResourceMap().values();
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            return Collections.emptyList();
        } finally {
            IOUtils.closeQuietly(stream);
        }
    }

    @Override
    public ServerResponse preProcess(HttpRequest request, ResourceMethodInvoker invoker) throws Failure, WebApplicationException {
        if (failedToInitialize) {
            LOG.error("UMA Protector failed to initialize.");
            return null;
        }

        String rpt = RptPreProcessInterceptor.getRpt(request.getHttpHeaders());
        if (UGLY_HARDCODE.equalsIgnoreCase(rpt)) {
            LOG.error("HARDCODE : Gerald, please fix ecommerce !");
            return null;
        }
        return interceptor.preProcess(request, invoker);
    }
}
