package org.gluu.oxd.licenser.server.ws;

import com.google.inject.Inject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gluu.oxd.license.client.Jackson;
import org.gluu.oxd.license.client.js.InfoMacAddressRequest;
import org.gluu.oxd.license.client.js.LdapLicenseId;
import org.gluu.oxd.license.client.js.LicenseMetadata;
import org.gluu.oxd.license.client.js.event.LdapClientEvent;
import org.gluu.oxd.licenser.server.model.LicenseMetadataResponse;
import org.gluu.oxd.licenser.server.service.ErrorService;
import org.gluu.oxd.licenser.server.service.LicenseIdService;
import org.gluu.oxd.licenser.server.service.StatisticService;
import org.gluu.oxd.licenser.server.service.ValidationService;
import org.json.JSONObject;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Returns metadata for given license id.
 * <p/>
 * Created by yuriy on 8/15/2015.
 */
@Path("/rest")
public class MetadataWS {

    private static final Logger LOG = LoggerFactory.getLogger(MetadataWS.class);

    @Inject
    ValidationService validationService;
    @Inject
    ErrorService errorService;
    @Inject
    LicenseIdService licenseIdService;
    @Inject
    StatisticService statisticService;

    @GET
    @Path("/metadata")
    @Produces({MediaType.APPLICATION_JSON})
    public Response generateGet(@QueryParam("licenseId") String licenseId) {
        LOG.debug("Metadata request, licenseId: " + licenseId);
        String json = Jackson.asJsonSilently(new LicenseMetadataResponse(metadata(licenseId)));
        return Response.ok().entity(json).build();
    }

    @PUT
    @Path("/metadata")
    @Produces({MediaType.APPLICATION_JSON})
    public Response update(LicenseMetadata metadata) {
        try {
            LOG.debug("Updating licenseId: " + metadata.getLicenseId() + ", metadata: " + metadata);
            validationService.validate(metadata);

            LdapLicenseId licenseId = validationService.getLicenseId(metadata.getLicenseId());
            licenseId.setMetadataAsObject(metadata);
            licenseId.setMetadata(Jackson.asJsonSilently(metadata));
            licenseIdService.merge(licenseId);

            LOG.debug("LicenseId: " + metadata.getLicenseId() + " updated.");
            return Response.ok().build();
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            throw new WebApplicationException(ErrorService.response(e.getMessage()));
        }
    }

    public LicenseMetadata metadata(String licenseIdStr) {
        try {
            LdapLicenseId licenseId = validationService.getLicenseId(licenseIdStr);
            LicenseMetadata metadata = Jackson.createJsonMapper().readValue(licenseId.getMetadata(), LicenseMetadata.class);
            metadata.setLicenseId(licenseId.getLicenseId());
            return metadata;
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            throw new WebApplicationException(ErrorService.response(e.getMessage()));
        }
    }

    @POST
    @Path("/info/macAddress")
    @Produces({MediaType.APPLICATION_JSON})
    @Consumes({MediaType.APPLICATION_JSON})
    public Response infoMacAddress(InfoMacAddressRequest request) {
        try {
            LOG.debug("Info request: " + request);
            validationService.getLicenseId(request.getLicenseId());

            final List<LdapClientEvent> all = statisticService.getAll(request.getLicenseId());
            final Map<String, List<LdapClientEvent>> map = new HashMap<>();

            for (LdapClientEvent event : all) {

                if (!request.getMacAddresses().contains(event.getMacAddress())) {
                    continue;
                }

                List<LdapClientEvent> value = map.get(event.getMacAddress());
                if (value == null) {
                    value = new ArrayList<>();
                    map.put(event.getMacAddress(), value);
                }
                value.add(event);
            }

            JSONObject response = new JSONObject();
            for (Map.Entry<String, List<LdapClientEvent>> entry : map.entrySet()) {
                final LdapClientEvent min = Collections.min(entry.getValue(), new Comparator<LdapClientEvent>() {
                    @Override
                    public int compare(LdapClientEvent o1, LdapClientEvent o2) {
                        return o1.getCreationDate().compareTo(o2.getCreationDate());
                    }
                });

                JSONObject properties = new JSONObject();
                properties.put("creation_date", min.getCreationDate());

                response.put(entry.getKey(), properties);
            }

            return Response.ok().entity(response.toString(2)).build();
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            throw new WebApplicationException(ErrorService.response(e.getMessage()));
        }
    }

}
