package org.gluu.oxd.licenser.server.service;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.inject.Inject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gluu.oxd.license.client.js.event.LdapLicenseGeneratedEvent;
import org.json.JSONObject;

import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Yuriy Zabrovarnyy
 */

public class LicenseGeneratedStatService {

    private static final Logger LOG = LoggerFactory.getLogger(LicenseGeneratedStatService.class);

    @Inject
    ValidationService validationService;
    @Inject
    StatisticService statisticService;

    public String monthlyStatistic(String licenseId) {
        validationService.getLicenseId(licenseId);

        try {
            List<LdapLicenseGeneratedEvent> entities = statisticService.getAllLicenseGenerated(licenseId);
            Map<String, List<LdapLicenseGeneratedEvent>> map = constructMonthlyMap(entities);

            JSONObject monthlyStatistic = new JSONObject();

            for (Map.Entry<String, List<LdapLicenseGeneratedEvent>> entity : map.entrySet()) {
                JSONObject stat = new JSONObject();
                stat.put("license_generated_count", entity.getValue().size());
                stat.put("mac_address", macAddressMap(entity.getValue()));

                monthlyStatistic.put(entity.getKey(), stat);
            }

            JSONObject wrapper = new JSONObject();
            wrapper.put("monthly_statistic", monthlyStatistic);
            wrapper.put("total_generated_licenses", entities.size());
            return wrapper.toString(2);
        } catch (Exception e) {
            LOG.error("Failed to construct statistic for license_id: " + licenseId, e);
            return "{\"error\":\"Failed to construct statistic for license_id: " + licenseId + "\"}";
        }
    }

    private Map<String, Integer> macAddressMap(List<LdapLicenseGeneratedEvent> value) {
        Map<String, Integer> map = Maps.newHashMap();
        for (LdapLicenseGeneratedEvent v : value) {
            if (Strings.isNullOrEmpty(v.getMacAddress())) {
                continue;
            }

            Integer counter = map.get(v.getMacAddress());
            if (counter == null) {
                map.put(v.getMacAddress(), 1);
            } else {
                counter++;
                map.put(v.getMacAddress(), counter);
            }
        }
        return map;
    }

    private Map<String, List<LdapLicenseGeneratedEvent>> constructMonthlyMap(List<LdapLicenseGeneratedEvent> entities) {
        Map<String, List<LdapLicenseGeneratedEvent>> map = Maps.newTreeMap();

        for (LdapLicenseGeneratedEvent s : entities) {
            String key = yearAndMonth(s.getCreationDate());
            List<LdapLicenseGeneratedEvent> monthList = map.get(key);
            if (monthList == null) {
                monthList = Lists.newArrayList();
                map.put(key, monthList);
            }
            monthList.add(s);
        }
        return map;
    }

    private String yearAndMonth(Date date) {
        Calendar creationDate = Calendar.getInstance();
        creationDate.setTime(date);

        int month = creationDate.get(Calendar.MONTH) + 1;
        int year = creationDate.get(Calendar.YEAR);
        return year + "-" + month;
    }

    public String lastHours(String licenseId, int hours) {
        validationService.getLicenseId(licenseId);
        ValidationService.validateHours(hours);

        try {
            List<LdapLicenseGeneratedEvent> filtered = StatisticService.filter(statisticService.getAllLicenseGenerated(licenseId), StatisticService.<LdapLicenseGeneratedEvent>timePredicate(hours));

            Map<String, Integer> macAddressToCounter = new HashMap<>();
            for (LdapLicenseGeneratedEvent item : filtered) {
                Integer counter = macAddressToCounter.get(item.getMacAddress());
                if (counter == null) {
                    counter = 1;
                } else {
                    counter++;
                }

                //LOG.trace("macAddressToCounter put, mac: " + item.getMacAddress() + ", counter: " + counter);
                macAddressToCounter.put(item.getMacAddress(), counter);
            }

            LOG.trace("macAddressToCounter size: " + macAddressToCounter.size());

            JSONObject lastHoursStatistic = new JSONObject();

            for (Map.Entry<String, Integer> entity : macAddressToCounter.entrySet()) {
                //                JSONObject stat = new JSONObject();
                //                stat.put("mac_address", entity.getKey());
                //                stat.put("count", entity.getValue());

                lastHoursStatistic.put(entity.getKey(), entity.getValue());
            }

            JSONObject wrapper = new JSONObject();
            wrapper.put("statistic", lastHoursStatistic);
            wrapper.put("total_generated_licenses", filtered.size());
            return wrapper.toString(2);
        } catch (Exception e) {
            LOG.error("Failed to construct statistic for license_id: " + licenseId, e);
            return "{\"error\":\"Failed to construct statistic for license_id: " + licenseId + "\"}";
        }
    }
}
