package org.gluu.oxd.license.admin.server;

import com.google.common.collect.Lists;
import com.google.gwt.thirdparty.guava.common.base.Strings;
import com.google.gwt.user.server.rpc.RemoteServiceServlet;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import org.jboss.resteasy.client.ClientResponseFailure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gluu.oxd.license.admin.client.service.AdminService;
import org.gluu.oxd.license.admin.shared.IdTokenValidationResult;
import org.gluu.oxd.license.client.ClientUtils;
import org.gluu.oxd.license.client.GenerateWS;
import org.gluu.oxd.license.client.Jackson;
import org.gluu.oxd.license.client.LicenseClient;
import org.gluu.oxd.license.client.js.Configuration;
import org.gluu.oxd.license.client.js.LdapLicenseCrypt;
import org.gluu.oxd.license.client.js.LdapLicenseId;
import org.gluu.oxd.license.client.js.LicenseIdItem;
import org.gluu.oxd.license.client.js.LicenseMetadata;
import org.gluu.oxd.licenser.server.service.LicenseCryptService;
import org.gluu.oxd.licenser.server.service.LicenseIdService;

import java.util.Collection;
import java.util.List;

/**
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 23/09/2014
 */

@Singleton
public class AdminServiceImpl extends RemoteServiceServlet implements AdminService {

    private static final Logger LOG = LoggerFactory.getLogger(AdminServiceImpl.class);

    @Inject
    LicenseCryptService licenseCryptService;
    @Inject
    LicenseIdService licenseIdService;
    @Inject
    Configuration conf;

    private void validateIdToken(String idToken) {
        IdTokenValidationResult result = hasAccess(idToken);
        if (result == IdTokenValidationResult.ACCESS_GRANTED) {
            return;
        }
        throw new IllegalArgumentException("Failed to validate id_token: " + idToken);
    }

    @Override
    public Configuration getConfiguration(String idToken) {
        //validateIdToken(idToken);
        return conf.copyForUI();
    }

    @Override
    public LdapLicenseCrypt generate(String idToken) {
        validateIdToken(idToken);
        return licenseCryptService.generate();
    }

    @Override
    public List<LdapLicenseId> generateLicenseIds(String idToken, int count, LdapLicenseCrypt licenseCrypt, LicenseMetadata metadata) {
        validateIdToken(idToken);
        return licenseIdService.generateLicenseIdsWithPersistence(count, licenseCrypt, metadata);
    }

    @Override
    public List<LdapLicenseId> loadLicenseIdsByCrypt(String idToken, LdapLicenseCrypt licenseCrypt) {
        validateIdToken(idToken);
        return licenseIdService.getByCryptDn(licenseCrypt.getDn());
    }

    @Override
    public LdapLicenseCrypt getLicenseCrypt(String idToken, String dn) {
        validateIdToken(idToken);
        return licenseCryptService.get(dn);
    }

    @Override
    public void save(String idToken, LdapLicenseId entity) {
        validateIdToken(idToken);
        final LicenseMetadata metadataAsObject = entity.getMetadataAsObject();
        if (metadataAsObject != null) {
            entity.setMetadata(Jackson.asJsonSilently(metadataAsObject));
        }

        if (Strings.isNullOrEmpty(entity.getDn())) {
            licenseIdService.save(entity);
        } else {
            licenseIdService.merge(entity);
        }
    }

    @Override
    public LdapLicenseId getLicenseId(String idToken, String licenseId) {
        validateIdToken(idToken);
        return licenseIdService.getById(licenseId);
    }

    @Override
    public boolean isGenerationApiProtected(String idToken) {
        validateIdToken(idToken);
        return isLicenseApiProtected();
    }

    private static boolean isLicenseApiProtected() {
        try {
            List<LicenseIdItem> list = generateWS().generateLicenseId(1, "", Tester.testMetadata());
            if (!list.isEmpty()) {
                LOG.error("SEVERE ERROR : License ID generation endpoint is not protected!");
            }
            return list.isEmpty();
        } catch (ClientResponseFailure e) {
            int status = e.getResponse().getStatus();
            return status == 403 || status == 401;
        }
    }

    private static GenerateWS generateWS() {
        return LicenseClient.generateWs(ClientUtils.LICENSE_SERVER_ENDPOINT, ClientUtils.executor());
    }

    public static void main(String[] args) {
        System.out.println(isLicenseApiProtected());
    }

    @Override
    public void save(String idToken, LdapLicenseCrypt entity) {
        validateIdToken(idToken);
        if (Strings.isNullOrEmpty(entity.getDn())) {
            licenseCryptService.save(entity);
        } else {
            licenseCryptService.merge(entity);
        }
    }

    @Override
    public void remove(String idToken, LdapLicenseCrypt entity) {
        validateIdToken(idToken);
        licenseCryptService.remove(entity);
        remove(idToken, loadLicenseIdsByCrypt(idToken, entity));
    }

    @Override
    public void remove(String idToken, Collection<LdapLicenseId> entities) {
        validateIdToken(idToken);
        for (LdapLicenseId entry : entities) {
            licenseIdService.remove(entry);
        }
    }

    @Override
    public List<LdapLicenseCrypt> getAllLicenseCryptObjects(String idToken) {
        validateIdToken(idToken);
        try {
            return licenseCryptService.getAll();
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            return Lists.newArrayList();
        }
    }

    @Override
    public IdTokenValidationResult hasAccess(String idToken) {
        return new IdTokenValidator(conf.getUserAccess()).hasAccess(idToken);
    }
}
