package org.gluu.oxd.license.admin.client.dialogs;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.uibinder.client.UiBinder;
import com.google.gwt.uibinder.client.UiField;
import com.google.gwt.user.client.ui.*;
import com.google.gwt.user.datepicker.client.DateBox;
import org.gluu.oxd.license.admin.client.Admin;
import org.gluu.oxd.license.admin.client.Framework;
import org.gluu.oxd.license.client.js.LdapLicenseId;
import org.gluu.oxd.license.client.js.LicenseMetadata;
import org.gluu.oxd.license.client.js.Product;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Yuriy Zabrovarnyy
 * @version 0.9, 06/10/2014
 */

public class LicenseIdMetadataDialog {

    private static MyUiBinder uiBinder = GWT.create(MyUiBinder.class);

    interface MyUiBinder extends UiBinder<Widget, LicenseIdMetadataDialog> {
    }

    private final DialogBox dialog;

    @UiField
    VerticalPanel dialogContent;
    @UiField
    HTML errorMessage;
    @UiField
    Button okButton;
    @UiField
    Button closeButton;
    @UiField
    TextBox numberOfLicenseIds;
    @UiField
    HTML numberOfLicenseIdsLabel;
    @UiField
    TextBox licenseName;
    @UiField
    ListBox product;
    @UiField
    DateBox expirationDate;
    @UiField
    TextBox licenseCountLimit;
    @UiField
    TextBox userCountLimit;
    @UiField
    CheckBox active;
    @UiField
    CheckBox autoupdate;

    private final LdapLicenseId licenseId;
    private final boolean isEditMode;

    public LicenseIdMetadataDialog(LdapLicenseId licenseId) {
        uiBinder.createAndBindUi(this);

        this.licenseId = licenseId;
        this.isEditMode = licenseId != null;

        dialog = Framework.createDialogBox("License Id configuration");
        dialog.setWidget(dialogContent);

        numberOfLicenseIds.setValue("1");
        licenseCountLimit.setValue(Integer.toString(LicenseMetadata.DEFAULT_LICENSE_COUNT_LIMIT));
        userCountLimit.setValue("1");

        for (Product p : Product.values()) {
            this.product.addItem(p.getValue(), p.getValue());
        }

        setEditMode();

        closeButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                dialog.hide();
            }
        });
        okButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                if (validate()) {
                    dialog.hide();
                    onOk();
                }
            }
        });
    }

    private void setEditMode() {
        if (!isEditMode) {
            return;
        }

        numberOfLicenseIds.setVisible(false);
        numberOfLicenseIdsLabel.setVisible(false);

        final LicenseMetadata metadataAsObject = licenseId.getMetadataAsObject();
        if (metadataAsObject != null) {
            active.setValue(metadataAsObject.getActive());
            autoupdate.setValue(metadataAsObject.getAutoupdate());
            licenseName.setValue(metadataAsObject.getLicenseName());
            licenseCountLimit.setValue(Integer.toString(metadataAsObject.getLicenseCountLimit()));
            userCountLimit.setValue(Integer.toString(metadataAsObject.getUserCountLimit()));
            expirationDate.setValue(metadataAsObject.getExpirationDate());

            selectProductItem(metadataAsObject.getProduct()); // it is only for backwards compatibility with previous version (see https://github.com/GluuFederation/oxLicense/issues/19)
            for (String p : metadataAsObject.getProducts()) {
                selectProductItem(p);
            }
        }
    }

    private void selectProductItem(String productItem) {
        if (!Admin.isEmpty(productItem)) {
            final int index = productIndex(productItem);
            if (index != -1 ) {
                product.setItemSelected(index, true);
            }
        }
    }

    private int productIndex(String product) {
        for (int i = 0; i < this.product.getItemCount(); i++) {
            if (this.product.getValue(i).equals(product)) {
                return i;
            }
        }
        return -1;
    }

    private void showError(String message) {
        errorMessage.setVisible(true);
        errorMessage.setHTML("<span style='color:red;'>" + message + "</span>");
    }

    public List<String> getSelectedProducts() {
        List<String> products = new ArrayList<String>();
        for (int i = 0; i < product.getItemCount(); i++) {
            if (product.isItemSelected(i)) {
                products.add(product.getValue(i));
            }
        }
        return products;
    }

    private boolean validate() {
        errorMessage.setVisible(false);

        final Integer numberOfLicenses = numberOfLicenses();
        final Integer licenseCountLimit = licenseCountLimit();
        final Integer userCountLimit = userCountLimit();

        if ((numberOfLicenses == null || numberOfLicenses <= 0) && !isEditMode) {
            showError("Unable to parse number of licenses. Please enter integer more then zero.");
            return false;
        }
        if (licenseCountLimit == null || licenseCountLimit < 0) {
            showError("Unable to parse number of license count limit.");
            return false;
        }
        if (userCountLimit == null) {
            showError("Unable to parse number of user count limit.");
            return false;
        }
        if (getSelectedProducts().isEmpty()) {
            showError("Product list is blank. Please select products. Supported products: " + Product.supportedProductsString());
            return false;
        }
        if (expirationDate.getValue() == null) {
            showError("Expiration date is not set.");
            return false;
        }

        final String licenseName = this.licenseName.getValue();
        if (licenseName == null || licenseName.isEmpty()) {
            showError("Please enter name for license.");
            return false;
        }

        return true;
    }

    public LicenseMetadata licenseMetadata() {
        LicenseMetadata licenseMetadata = new LicenseMetadata();
        licenseMetadata.setActive(active.getValue());
        licenseMetadata.setAutoupdate(autoupdate.getValue());
        licenseMetadata.setProducts(getSelectedProducts());
        licenseMetadata.setLicenseName(licenseName.getValue());
        licenseMetadata.setLicenseCountLimit(licenseCountLimit());
        licenseMetadata.setUserCountLimit(userCountLimit());
        licenseMetadata.setExpirationDate(expirationDate.getValue());

        return licenseMetadata;
    }

    public Integer numberOfLicenses() {
        return Admin.parse(numberOfLicenseIds.getValue());
    }

    public Integer licenseCountLimit() {
        return Admin.parse(licenseCountLimit.getValue());
    }

    public Integer userCountLimit() {
        return Admin.parse(userCountLimit.getValue());
    }

    public void onOk() {
    }

    public void setTitle(String title) {
        dialog.setText(title);
        dialog.setTitle(title);
    }

    public void show() {
        dialog.center();
        dialog.show();
    }

    public LdapLicenseId getLicenseId() {
        if (isEditMode) {
            licenseId.setMetadataAsObject(licenseMetadata());
        }
        return licenseId;
    }
}
