/*
 * Decompiled with CFR 0.152.
 */
package org.gluu.oxauth.cert.validation;

import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.NoSuchProviderException;
import java.security.cert.CRLException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509CRLEntry;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import javax.security.auth.x500.X500Principal;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.BoundedInputStream;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERIA5String;
import org.bouncycastle.asn1.x509.CRLDistPoint;
import org.bouncycastle.asn1.x509.DistributionPoint;
import org.bouncycastle.asn1.x509.DistributionPointName;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.X509Extensions;
import org.bouncycastle.x509.NoSuchParserException;
import org.bouncycastle.x509.util.StreamParsingException;
import org.gluu.oxauth.cert.validation.CertificateVerifier;
import org.gluu.oxauth.cert.validation.model.ValidationStatus;
import org.gluu.util.security.SecurityProviderUtility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CRLCertificateVerifier
implements CertificateVerifier {
    private static final Logger log = LoggerFactory.getLogger(CRLCertificateVerifier.class);
    private int maxCrlSize;
    private LoadingCache<String, X509CRL> crlCache;

    public CRLCertificateVerifier(int maxCrlSize) {
        SecurityProviderUtility.installBCProvider((boolean)true);
        this.maxCrlSize = maxCrlSize;
        CacheLoader<String, X509CRL> checkedLoader = new CacheLoader<String, X509CRL>(){

            public X509CRL load(String crlURL) throws CertificateException, CRLException, NoSuchProviderException, NoSuchParserException, StreamParsingException, MalformedURLException, IOException, ExecutionException {
                X509CRL result = CRLCertificateVerifier.this.requestCRL(crlURL);
                Preconditions.checkNotNull((Object)result);
                return result;
            }
        };
        this.crlCache = CacheBuilder.newBuilder().maximumSize(10L).expireAfterWrite(60L, TimeUnit.MINUTES).build((CacheLoader)checkedLoader);
    }

    @Override
    public ValidationStatus validate(X509Certificate certificate, List<X509Certificate> issuers, Date validationDate) {
        X509Certificate issuer = issuers.get(0);
        ValidationStatus status = new ValidationStatus(certificate, issuer, validationDate, ValidationStatus.ValidatorSourceType.CRL, ValidationStatus.CertificateValidity.UNKNOWN);
        try {
            X500Principal subjectX500Principal = certificate.getSubjectX500Principal();
            String crlURL = this.getCrlUri(certificate);
            if (crlURL == null) {
                log.error("CRL's URL for '" + subjectX500Principal + "' is empty");
                return status;
            }
            log.debug("CRL's URL for '" + subjectX500Principal + "' is '" + crlURL + "'");
            X509CRL x509crl = this.getCrl(crlURL);
            if (!this.validateCRL(x509crl, certificate, issuer, validationDate)) {
                log.error("The CRL is not valid!");
                status.setValidity(ValidationStatus.CertificateValidity.INVALID);
                return status;
            }
            X509CRLEntry crlEntry = x509crl.getRevokedCertificate(certificate.getSerialNumber());
            if (crlEntry == null) {
                log.debug("CRL status is valid for '" + subjectX500Principal + "'");
                status.setValidity(ValidationStatus.CertificateValidity.VALID);
            } else if (crlEntry.getRevocationDate().after(validationDate)) {
                log.warn("CRL revocation time after the validation date, the certificate '" + subjectX500Principal + "' was valid at " + validationDate);
                status.setRevocationObjectIssuingTime(x509crl.getThisUpdate());
                status.setValidity(ValidationStatus.CertificateValidity.VALID);
            } else {
                log.info("CRL for certificate '" + subjectX500Principal + "' is revoked since " + crlEntry.getRevocationDate());
                status.setRevocationObjectIssuingTime(x509crl.getThisUpdate());
                status.setRevocationDate(crlEntry.getRevocationDate());
                status.setValidity(ValidationStatus.CertificateValidity.REVOKED);
            }
        }
        catch (Exception ex) {
            log.error("CRL exception: ", (Throwable)ex);
        }
        return status;
    }

    private boolean validateCRL(X509CRL x509crl, X509Certificate certificate, X509Certificate issuerCertificate, Date validationDate) {
        X500Principal subjectX500Principal = certificate.getSubjectX500Principal();
        if (x509crl == null) {
            log.error("No CRL found for certificate '" + subjectX500Principal + "'");
            return false;
        }
        if (log.isTraceEnabled()) {
            try {
                log.trace("CRL number: " + this.getCrlNumber(x509crl));
            }
            catch (IOException ex) {
                log.error("Failed to get CRL number", (Throwable)ex);
            }
        }
        if (!x509crl.getIssuerX500Principal().equals(issuerCertificate.getSubjectX500Principal())) {
            log.error("The CRL must be signed by the issuer '" + subjectX500Principal + "' but instead is signed by '" + x509crl.getIssuerX500Principal() + "'");
            return false;
        }
        try {
            x509crl.verify(issuerCertificate.getPublicKey());
        }
        catch (Exception ex) {
            log.error("The signature verification for CRL cannot be performed", (Throwable)ex);
            return false;
        }
        log.debug("CRL validationDate: " + validationDate);
        log.debug("CRL nextUpdate: " + x509crl.getThisUpdate());
        log.debug("CRL thisUpdate: " + x509crl.getNextUpdate());
        if (x509crl.getNextUpdate() != null && validationDate.after(x509crl.getNextUpdate())) {
            log.error("CRL is too old");
            return false;
        }
        if (issuerCertificate.getKeyUsage() == null) {
            log.error("There is no KeyUsage extension for certificate '" + subjectX500Principal + "'");
            return false;
        }
        if (!issuerCertificate.getKeyUsage()[6]) {
            log.error("cRLSign bit is not set for CRL certificate'" + subjectX500Principal + "'");
            return false;
        }
        return true;
    }

    private X509CRL getCrl(String url) throws CertificateException, CRLException, NoSuchProviderException, NoSuchParserException, StreamParsingException, MalformedURLException, IOException, ExecutionException {
        if (!url.startsWith("http://") && !url.startsWith("https://")) {
            log.error("It's possible to download CRL via HTTP and HTTPS only");
            return null;
        }
        String cacheKey = url.toLowerCase();
        X509CRL crl = (X509CRL)this.crlCache.get((Object)cacheKey);
        return crl;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public X509CRL requestCRL(String url) throws IOException, MalformedURLException, CertificateException, CRLException {
        block9: {
            HttpURLConnection con = (HttpURLConnection)new URL(url).openConnection();
            try {
                X509CRL x509CRL;
                con.setUseCaches(false);
                BoundedInputStream in = new BoundedInputStream(con.getInputStream(), (long)this.maxCrlSize);
                try {
                    CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
                    X509CRL crl = (X509CRL)certificateFactory.generateCRL((InputStream)in);
                    log.debug("CRL size: " + crl.getEncoded().length + " bytes");
                    x509CRL = crl;
                }
                catch (Throwable throwable) {
                    try {
                        IOUtils.closeQuietly((InputStream)in);
                        throw throwable;
                    }
                    catch (IOException ex) {
                        log.error("Failed to download CRL from '" + url + "'", (Throwable)ex);
                        break block9;
                    }
                }
                IOUtils.closeQuietly((InputStream)in);
                return x509CRL;
            }
            finally {
                if (con != null) {
                    con.disconnect();
                }
            }
        }
        return null;
    }

    private BigInteger getCrlNumber(X509CRL crl) throws IOException {
        byte[] crlNumberExtensionValue = crl.getExtensionValue(X509Extensions.CRLNumber.getId());
        if (crlNumberExtensionValue == null) {
            return null;
        }
        ASN1OctetString octetString = (ASN1OctetString)new ASN1InputStream((InputStream)new ByteArrayInputStream(crlNumberExtensionValue)).readObject();
        byte[] octets = octetString.getOctets();
        ASN1Integer integer = (ASN1Integer)new ASN1InputStream(octets).readObject();
        BigInteger crlNumber = integer.getPositiveValue();
        return crlNumber;
    }

    public String getCrlUri(X509Certificate certificate) throws IOException {
        DistributionPoint[] distributionPoints;
        ASN1Primitive obj;
        try {
            obj = CRLCertificateVerifier.getExtensionValue(certificate, Extension.cRLDistributionPoints.getId());
        }
        catch (IOException ex) {
            log.error("Failed to get CRL URL", (Throwable)ex);
            return null;
        }
        if (obj == null) {
            return null;
        }
        CRLDistPoint distPoint = CRLDistPoint.getInstance((Object)obj);
        for (DistributionPoint distributionPoint : distributionPoints = distPoint.getDistributionPoints()) {
            GeneralName[] names;
            DistributionPointName distributionPointName = distributionPoint.getDistributionPoint();
            if (0 != distributionPointName.getType()) continue;
            GeneralNames generalNames = (GeneralNames)distributionPointName.getName();
            for (GeneralName name : names = generalNames.getNames()) {
                if (name.getTagNo() != 6) continue;
                DERIA5String derStr = DERIA5String.getInstance((ASN1TaggedObject)((ASN1TaggedObject)name.toASN1Primitive()), (boolean)false);
                return derStr.getString();
            }
        }
        return null;
    }

    private static ASN1Primitive getExtensionValue(X509Certificate certificate, String oid) throws IOException {
        byte[] bytes = certificate.getExtensionValue(oid);
        if (bytes == null) {
            return null;
        }
        ASN1InputStream aIn = new ASN1InputStream((InputStream)new ByteArrayInputStream(bytes));
        ASN1OctetString octs = (ASN1OctetString)aIn.readObject();
        aIn = new ASN1InputStream((InputStream)new ByteArrayInputStream(octs.getOctets()));
        return aIn.readObject();
    }

    @Override
    public void destroy() {
        this.crlCache.cleanUp();
    }
}

