/*
 * Decompiled with CFR 0.152.
 */
package org.gluu.casa.plugins.cert.service;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.security.cert.CertificateFactory;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import org.gluu.casa.core.model.BasePerson;
import org.gluu.casa.core.model.IdentityPerson;
import org.gluu.casa.misc.Utils;
import org.gluu.casa.plugins.cert.model.CertPerson;
import org.gluu.casa.plugins.cert.model.Certificate;
import org.gluu.casa.plugins.cert.service.UserCertificateMatch;
import org.gluu.casa.plugins.cert.service.VerifierType;
import org.gluu.casa.service.IPersistenceService;
import org.gluu.oxauth.cert.fingerprint.FingerprintHelper;
import org.gluu.oxauth.cert.validation.CRLCertificateVerifier;
import org.gluu.oxauth.cert.validation.CertificateVerifier;
import org.gluu.oxauth.cert.validation.GenericCertificateVerifier;
import org.gluu.oxauth.cert.validation.OCSPCertificateVerifier;
import org.gluu.oxauth.cert.validation.PathCertificateVerifier;
import org.gluu.oxauth.cert.validation.model.ValidationStatus;
import org.gluu.oxauth.model.util.CertUtils;
import org.gluu.oxtrust.model.scim2.user.X509Certificate;
import org.gluu.search.filter.Filter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertService {
    private static final String ACR = "cert";
    private static final int DEFAULT_CRL_MAX_RESPONSE_SIZE = 0x500000;
    private static final String CERT_PREFIX = "cert:";
    private static CertService singleton;
    private Logger logger = LoggerFactory.getLogger(this.getClass());
    private ObjectMapper mapper = new ObjectMapper();
    private IPersistenceService persistenceService = (IPersistenceService)Utils.managedBean(IPersistenceService.class);
    private Map<String, String> scriptProperties;
    private int crlMaxResponseSize;
    private List<java.security.cert.X509Certificate> chainCerts;
    private boolean hasValidProperties;

    public static CertService getInstance() {
        if (singleton == null) {
            singleton = new CertService();
        }
        return singleton;
    }

    public boolean isHasValidProperties() {
        return this.hasValidProperties;
    }

    public void reloadConfiguration() {
        this.scriptProperties = this.persistenceService.getCustScriptConfigProperties(ACR);
        if (this.scriptProperties == null) {
            this.logger.warn("Config. properties for custom script '{}' could not be read!!.", (Object)ACR);
        } else {
            String paramName = "crl_max_response_size";
            try {
                this.crlMaxResponseSize = Integer.valueOf(this.scriptProperties.get(paramName));
            }
            catch (Exception e) {
                this.logger.error(e.getMessage());
                this.logger.warn("Using default value of {} for '{}'", (Object)0x500000, (Object)paramName);
            }
            paramName = "chain_cert_file_path";
            this.logger.info("Scanning cert chains specified in '{}' param...", (Object)paramName);
            try (InputStream is = Files.newInputStream(Paths.get(this.scriptProperties.get(paramName), new String[0]), new OpenOption[0]);){
                CertificateFactory cf = CertificateFactory.getInstance("X.509");
                this.chainCerts = cf.generateCertificates(is).stream().map(java.security.cert.X509Certificate.class::cast).collect(Collectors.toList());
                this.logger.info("{} certs loaded", (Object)this.chainCerts.size());
                this.hasValidProperties = true;
            }
            catch (Exception e) {
                this.logger.error(e.getMessage(), (Throwable)e);
            }
        }
    }

    public boolean validate(java.security.cert.X509Certificate cert) {
        boolean valid = true;
        Date date = new Date();
        this.logger.info("Validating certificate");
        for (VerifierType type : VerifierType.values()) {
            String param = this.scriptProperties.get(type.getParam());
            if (!Optional.ofNullable(param).map(Boolean::valueOf).orElse(false).booleanValue()) continue;
            CertificateVerifier verifier = null;
            this.logger.info("Applying '{}' validator", (Object)type);
            switch (type) {
                case GENERIC: {
                    verifier = new GenericCertificateVerifier();
                    break;
                }
                case PATH: {
                    verifier = new PathCertificateVerifier(true);
                    break;
                }
                case OSCP: {
                    verifier = new OCSPCertificateVerifier();
                    break;
                }
                case CRL: {
                    verifier = new CRLCertificateVerifier(this.crlMaxResponseSize);
                }
            }
            ValidationStatus.CertificateValidity validity = verifier.validate(cert, this.chainCerts, date).getValidity();
            if (validity.equals((Object)ValidationStatus.CertificateValidity.VALID)) continue;
            valid = false;
            this.logger.warn("Certificate validity is: {}", (Object)validity);
            break;
        }
        return valid;
    }

    public UserCertificateMatch processMatch(java.security.cert.X509Certificate certificate, String userInum, boolean enroll) {
        UserCertificateMatch status = null;
        try {
            this.logger.info("Matching certificate and user. Enrollment is {}", (Object)enroll);
            CertPerson person = (CertPerson)((Object)this.persistenceService.get(CertPerson.class, this.persistenceService.getPersonDn(userInum)));
            if (person == null) {
                status = UserCertificateMatch.UNKNOWN_USER;
            } else {
                this.logger.debug("Generating certificate fingerprint...");
                String fingerprint = this.getFingerPrint(certificate);
                String externalUid = String.format("%s%s", CERT_PREFIX, fingerprint);
                Filter filter = Filter.createEqualityFilter((String)"oxExternalUid", (Object)externalUid).multiValued();
                List people = this.persistenceService.find(BasePerson.class, this.persistenceService.getPeopleDn(), filter, 0, 1);
                if (people.size() > 0) {
                    status = userInum.equals(((BasePerson)people.get(0)).getInum()) ? (enroll ? UserCertificateMatch.CERT_ENROLLED_ALREADY : UserCertificateMatch.SUCCESS) : UserCertificateMatch.CERT_ENROLLED_OTHER_USER;
                } else if (enroll) {
                    this.logger.info("Associating presented cert to user");
                    ArrayList<String> oeuid = new ArrayList<String>(Optional.ofNullable(person.getOxExternalUid()).orElse(Collections.emptyList()));
                    oeuid.add(externalUid);
                    person.setOxExternalUid(oeuid);
                    status = UserCertificateMatch.SUCCESS;
                } else {
                    this.logger.info("Certificate not associated to an existing account yet");
                    status = UserCertificateMatch.CERT_NOT_RECOGNIZED;
                }
            }
            if (status.equals((Object)UserCertificateMatch.SUCCESS) || status.equals((Object)UserCertificateMatch.CERT_ENROLLED_ALREADY)) {
                this.updateUserX509Certificates(person, certificate);
                status = this.persistenceService.modify((Object)person) ? status : UserCertificateMatch.UNKNOWN_ERROR;
            }
            this.logger.info("Operation result is {}", (Object)status.toString());
        }
        catch (Exception e) {
            this.logger.error(e.getMessage(), (Throwable)e);
            status = UserCertificateMatch.UNKNOWN_ERROR;
        }
        return status;
    }

    public List<Certificate> getUserCerts(String userId) {
        List<Certificate> certs = new ArrayList<Certificate>();
        try {
            CertPerson person = (CertPerson)((Object)this.persistenceService.get(CertPerson.class, this.persistenceService.getPersonDn(userId)));
            List<X509Certificate> x509Certificates = this.getScimX509Certificates(Optional.ofNullable(person.getX509Certificates()).orElse(Collections.emptyList()));
            certs = person.getOxExternalUid().stream().filter(uid -> uid.startsWith(CERT_PREFIX)).map(uid -> this.getExtraCertsInfo((String)uid, x509Certificates)).collect(Collectors.toList());
        }
        catch (Exception e) {
            this.logger.error(e.getMessage(), (Throwable)e);
        }
        return certs;
    }

    public int getDevicesTotal(String userId) {
        int total = 0;
        try {
            IdentityPerson person = (IdentityPerson)this.persistenceService.get(IdentityPerson.class, this.persistenceService.getPersonDn(userId));
            total = (int)person.getOxExternalUid().stream().filter(uid -> uid.startsWith(CERT_PREFIX)).count();
        }
        catch (Exception e) {
            this.logger.error(e.getMessage(), (Throwable)e);
        }
        return total;
    }

    private Certificate getExtraCertsInfo(String externalUid, List<X509Certificate> scimCerts) {
        String fingerPrint = externalUid.replace(CERT_PREFIX, "");
        Certificate cert = new Certificate();
        cert.setFingerPrint(fingerPrint);
        for (X509Certificate sc : scimCerts) {
            try {
                java.security.cert.X509Certificate x509Certificate = CertUtils.x509CertificateFromPem((String)sc.getValue());
                if (!fingerPrint.equals(this.getFingerPrint(x509Certificate))) continue;
                Map<String, String> attributes = Arrays.stream(sc.getDisplay().split(",\\s*")).collect(Collectors.toMap(t -> t.substring(0, t.indexOf(61)).toLowerCase(), t -> t.substring(t.indexOf(61) + 1)));
                String cn = attributes.get("cn");
                String ou = attributes.getOrDefault("ou", "");
                String o = attributes.getOrDefault("o", "");
                if (Utils.isNotEmpty((String)ou)) {
                    if (Utils.isNotEmpty((String)o)) {
                        ou = ou + ", " + o;
                    }
                } else if (Utils.isNotEmpty((String)o)) {
                    ou = o;
                }
                String l = attributes.getOrDefault("l", "");
                String st = attributes.getOrDefault("st", "");
                String c = attributes.getOrDefault("c", "");
                cert.setCommonName(cn);
                cert.setOrganization(ou);
                cert.setLocation(String.format("%s %s %s", l, st, c).trim());
                cert.setFormattedName(cn + (Utils.isEmpty((String)ou) ? "" : String.format(" (%s)", ou)));
                long date = x509Certificate.getNotAfter().getTime();
                cert.setExpirationDate(date);
                cert.setExpired(date < System.currentTimeMillis());
                break;
            }
            catch (Exception e) {
                this.logger.error(e.getMessage());
            }
        }
        return cert;
    }

    public boolean removeFromUser(String fingerPrint, String userId) throws Exception {
        int i;
        CertPerson person = (CertPerson)((Object)this.persistenceService.get(CertPerson.class, this.persistenceService.getPersonDn(userId)));
        List stringCerts = Optional.ofNullable(person.getX509Certificates()).orElse(new ArrayList());
        List<X509Certificate> scimCerts = this.getScimX509Certificates(stringCerts);
        boolean found = false;
        for (i = 0; i < scimCerts.size() && !found; ++i) {
            found = this.getFingerPrint(CertUtils.x509CertificateFromPem((String)scimCerts.get(i).getValue())).equals(fingerPrint);
        }
        if (found) {
            person.getX509Certificates().remove(i - 1);
        }
        Optional<String> externalUid = person.getOxExternalUid().stream().filter(str -> str.equals(CERT_PREFIX + fingerPrint)).findFirst();
        externalUid.ifPresent(uid -> person.getOxExternalUid().remove(uid));
        return this.persistenceService.modify((Object)person);
    }

    private List<X509Certificate> getScimX509Certificates(List<String> scimStringCerts) {
        ArrayList<X509Certificate> scimCerts = new ArrayList<X509Certificate>();
        for (String scimCert : scimStringCerts) {
            try {
                scimCerts.add((X509Certificate)this.mapper.readValue(scimCert, X509Certificate.class));
            }
            catch (Exception e) {
                this.logger.error("Unable to convert value '{}' to expected SCIM format", (Object)scimCert);
                this.logger.error(e.getMessage());
            }
        }
        return scimCerts;
    }

    private void updateUserX509Certificates(CertPerson person, java.security.cert.X509Certificate certificate) {
        try {
            boolean match = false;
            String display = certificate.getSubjectX500Principal().getName();
            this.logger.info("Reading user's stored X509 certificates");
            List stringCerts = Optional.ofNullable(person.getX509Certificates()).orElse(new ArrayList());
            List<X509Certificate> scimCerts = this.getScimX509Certificates(stringCerts);
            for (X509Certificate scimCert : scimCerts) {
                String scimDisplay = scimCert.getDisplay();
                if (!Utils.isNotEmpty((String)scimDisplay) || !scimDisplay.equals(display)) continue;
                this.logger.debug("The certificate presented is already in user's profile");
                match = true;
                break;
            }
            if (!match) {
                X509Certificate scimX509Cert = new X509Certificate();
                byte[] DEREncoded = certificate.getEncoded();
                scimX509Cert.setValue(new String(Base64.getEncoder().encode(DEREncoded), StandardCharsets.UTF_8));
                scimX509Cert.setDisplay(display);
                this.logger.debug("Updating user's oxTrustx509Certificate attribute");
                stringCerts.add(this.mapper.writeValueAsString((Object)scimX509Cert));
                person.setX509Certificates(stringCerts);
            }
        }
        catch (Exception e) {
            this.logger.error(e.getMessage(), (Throwable)e);
        }
    }

    private String getFingerPrint(java.security.cert.X509Certificate certificate) throws Exception {
        return FingerprintHelper.getPublicKeySshFingerprint(certificate.getPublicKey());
    }

    private CertService() {
        this.reloadConfiguration();
    }
}

