/*
 * Decompiled with CFR 0.152.
 */
package org.forgerock.audit.handlers.csv;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.security.SignatureException;
import java.util.Arrays;
import java.util.Map;
import javax.crypto.SecretKey;
import org.forgerock.audit.handlers.csv.CsvSecureUtils;
import org.forgerock.audit.handlers.csv.HmacCalculator;
import org.forgerock.audit.secure.SecureStorage;
import org.forgerock.audit.secure.SecureStorageException;
import org.forgerock.util.encode.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.supercsv.io.CsvMapReader;
import org.supercsv.prefs.CsvPreference;

class CsvSecureVerifier {
    private static final Logger logger = LoggerFactory.getLogger(CsvSecureVerifier.class);
    private File csvFile;
    private final CsvPreference csvPreference;
    private final HmacCalculator hmacCalculator;
    private final SecureStorage secureStorage;
    private String lastHMAC;
    private byte[] lastSignature;
    private String[] headers;

    public CsvSecureVerifier(File csvFile, CsvPreference csvPreference, SecureStorage secureStorage) {
        this.csvFile = csvFile;
        this.csvPreference = csvPreference;
        this.secureStorage = secureStorage;
        try {
            SecretKey initialKey = secureStorage.readInitialKey();
            if (initialKey == null) {
                throw new IllegalStateException("Expecting to find an initial key into the keystore.");
            }
            this.hmacCalculator = new HmacCalculator("HmacSHA256");
            this.hmacCalculator.setCurrentKey(initialKey.getEncoded());
        }
        catch (SecureStorageException e) {
            throw new IllegalStateException(e);
        }
    }

    public VerificationResult verify() throws IOException {
        boolean lastRowWasSigned = false;
        try (CsvMapReader csvReader = this.newBufferedCsvMapReader();){
            Map values;
            Object[] header = csvReader.getHeader(true);
            int checkCount = 0;
            for (String verificationResult : header) {
                if (!"HMAC".equals(verificationResult) && !"SIGNATURE".equals(verificationResult)) continue;
                ++checkCount;
            }
            if (!"HMAC".equals(header[header.length - 2]) || !"SIGNATURE".equals(header[header.length - 1])) {
                String msg = "Found only " + checkCount + " checked headers from : " + Arrays.toString(header);
                logger.debug(msg);
                VerificationResult len$ = this.newVerificationFailureResult(msg);
                return len$;
            }
            this.headers = new String[header.length - 2];
            System.arraycopy(header, 0, this.headers, 0, this.headers.length);
            while ((values = csvReader.read((String[])header)) != null) {
                logger.trace("Verifying row {}", (Object)csvReader.getRowNumber());
                lastRowWasSigned = false;
                String encodedSign = (String)values.get("SIGNATURE");
                if (encodedSign != null) {
                    if (csvReader.getRowNumber() == 2) {
                        this.lastSignature = Base64.decode((String)encodedSign);
                        continue;
                    }
                    if (!this.verifySignature(encodedSign)) {
                        String msg = "The signature at row " + csvReader.getRowNumber() + " is not correct.";
                        logger.trace(msg);
                        VerificationResult verificationResult = this.newVerificationFailureResult(msg);
                        return verificationResult;
                    }
                    logger.trace("The signature at row {} is correct.", (Object)csvReader.getRowNumber());
                    lastRowWasSigned = true;
                    continue;
                }
                if (!this.verifyHMAC(values, (String[])header)) {
                    String msg = "The HMac at row " + csvReader.getRowNumber() + " is not correct.";
                    logger.trace(msg);
                    VerificationResult verificationResult = this.newVerificationFailureResult(msg);
                    return verificationResult;
                }
                logger.trace("The HMac at row {} is correct.", (Object)csvReader.getRowNumber());
            }
        }
        try {
            SecretKey currentKey = this.secureStorage.readCurrentKey();
            if (currentKey != null) {
                boolean keysMatch = Arrays.equals(this.hmacCalculator.getCurrentKey().getEncoded(), currentKey.getEncoded());
                logger.trace("keysMatch={}, lastRowWasSigned={}", (Object)keysMatch, (Object)lastRowWasSigned);
                if (!keysMatch) {
                    return this.newVerificationFailureResult("Final HMAC key doesn't match expected value");
                }
                if (!lastRowWasSigned) {
                    return this.newVerificationFailureResult("Missing final signature");
                }
                return this.newVerificationSuccessResult();
            }
            logger.trace("currentKey is null");
            return this.newVerificationFailureResult("Final HMAC key is null");
        }
        catch (SecureStorageException ex) {
            throw new IOException(ex);
        }
    }

    private CsvMapReader newBufferedCsvMapReader() throws FileNotFoundException {
        return new CsvMapReader((Reader)new BufferedReader(new FileReader(this.csvFile)), this.csvPreference);
    }

    private VerificationResult newVerificationFailureResult(String msg) {
        return new VerificationResult(this.csvFile, false, msg);
    }

    private VerificationResult newVerificationSuccessResult() {
        return new VerificationResult(this.csvFile, true, "");
    }

    private boolean verifyHMAC(Map<String, String> values, String[] header) throws IOException {
        try {
            String actualHMAC = values.get("HMAC");
            String expectedHMAC = this.hmacCalculator.calculate(CsvSecureUtils.dataToSign(logger, values, this.dropExtraHeaders(header)));
            if (!actualHMAC.equals(expectedHMAC)) {
                logger.trace("The HMAC is not valid. Expected : {} Found : {}", (Object)expectedHMAC, (Object)actualHMAC);
                return false;
            }
            this.lastHMAC = actualHMAC;
            return true;
        }
        catch (SignatureException ex) {
            logger.error(ex.getMessage(), (Throwable)ex);
            throw new IOException(ex);
        }
    }

    private boolean verifySignature(String encodedSign) throws IOException {
        try {
            byte[] signature = Base64.decode((String)encodedSign);
            boolean verify = this.secureStorage.verify(CsvSecureUtils.dataToSign(this.lastSignature, this.lastHMAC), signature);
            if (!verify) {
                logger.trace("The signature does not match the expecting one.");
                return false;
            }
            this.lastSignature = signature;
            return true;
        }
        catch (SecureStorageException ex) {
            logger.error(ex.getMessage(), (Throwable)ex);
            throw new IOException(ex);
        }
    }

    private String[] dropExtraHeaders(String ... header) {
        return Arrays.copyOf(header, header.length - 2);
    }

    public String[] getHeaders() {
        return this.headers;
    }

    public String getLastHMAC() {
        return this.lastHMAC;
    }

    public byte[] getLastSignature() {
        return this.lastSignature;
    }

    static final class VerificationResult {
        private final File archiveFile;
        private final boolean passedVerification;
        private final String failureReason;

        VerificationResult(File archiveFile, boolean passedVerification, String message) {
            this.archiveFile = archiveFile;
            this.passedVerification = passedVerification;
            this.failureReason = message;
        }

        public File getArchiveFile() {
            return this.archiveFile;
        }

        public boolean hasPassedVerification() {
            return this.passedVerification;
        }

        public String getFailureReason() {
            return this.failureReason;
        }
    }
}

