/*
 * The contents of this file are subject to the terms of the Common Development and
 * Distribution License (the License). You may not use this file except in compliance with the
 * License.
 *
 * You can obtain a copy of the License at legal/CDDLv1.0.txt. See the License for the
 * specific language governing permission and limitations under the License.
 *
 * When distributing Covered Software, include this CDDL Header Notice in each file and include
 * the License file at legal/CDDLv1.0.txt. If applicable, add the following below the CDDL
 * Header, with the fields enclosed by brackets [] replaced by your own identifying
 * information: "Portions Copyright [year] [name of copyright owner]".
 *
 * Copyright 2006-2008 Sun Microsystems, Inc.
 * Portions Copyright 2011-2014 ForgeRock AS.
 */

package org.opends.quicksetup.ui;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.text.JTextComponent;

import org.opends.quicksetup.event.MinimumSizeComponentListener;
import org.opends.quicksetup.util.WebBrowserException;
import org.forgerock.i18n.LocalizableMessage;
import static org.opends.messages.QuickSetupMessages.*;

/**
 * This class is a dialog that appears when we could not launch the user web
 * browser after the user clicked on a link.
 *
 * The dialog displays the URL to be displayed and provides a 'Copy URL' button
 * to copy it to the system clipboard.  This way (even if not ideal) the user
 * can view the contents of the URL we display by pasting manually the URL
 * in his/her browser.
 *
 */
public class WebBrowserErrorDialog extends JDialog
{
  private static final long serialVersionUID = 1063837373763193941L;

  private JFrame parent;

  private String url;

  /**
   * Constructor of the WebBrowserErrorDialog.
   * @param parent the parent frame for this dialog.
   * @param ex the WebBrowserException.
   */
  public WebBrowserErrorDialog(JFrame parent, WebBrowserException ex)
  {
    super(parent);
    setTitle(INFO_ERROR_BROWSER_DISPLAY_TITLE.get().toString());
    this.parent = parent;
    this.url = ex.getUrl();
    getContentPane().add(createPanel());
  }

  /**
   * Packs and displays this dialog.
   *
   */
  public void packAndShow()
  {
    pack();
    int minWidth = (int) getPreferredSize().getWidth();
    int minHeight = (int) getPreferredSize().getHeight();
    addComponentListener(new MinimumSizeComponentListener(this,
        minWidth, minHeight));
    Utilities.centerOnComponent(this, parent);
    setVisible(true);
  }

  /**
   * Creates and returns the panel of the dialog.
   * @return the panel of the dialog.
   */
  private JPanel createPanel()
  {
    JPanel p1 = new JPanel(new GridBagLayout());
    p1.setBackground(UIFactory.CURRENT_STEP_PANEL_BACKGROUND);
    p1.setBorder(UIFactory.DIALOG_PANEL_BORDER);
    GridBagConstraints gbc = new GridBagConstraints();
    gbc.gridwidth = 3;
    gbc.anchor = GridBagConstraints.NORTHWEST;
    gbc.insets = UIFactory.getCurrentStepPanelInsets();
    p1.add(UIFactory.makeJLabel(UIFactory.IconType.WARNING_LARGE, null,
        UIFactory.TextStyle.NO_STYLE), gbc);
    gbc.weightx = 1.0;
    gbc.gridwidth = GridBagConstraints.RELATIVE;
    Insets pInsets = UIFactory.getCurrentStepPanelInsets();
    gbc.insets.left = 0;
    gbc.fill = GridBagConstraints.BOTH;
    LocalizableMessage msg = INFO_ERROR_BROWSER_DISPLAY_MSG.get(url);
    JTextComponent tf =
        UIFactory.makeHtmlPane(msg,
            UIFactory.ERROR_DIALOG_FONT);
    tf.setOpaque(false);
    tf.setEditable(false);
    p1.add(tf, gbc);

    gbc.weightx = 0.0;
    gbc.gridwidth = GridBagConstraints.REMAINDER;
    JButton copyButton =
        UIFactory.makeJButton(INFO_ERROR_BROWSER_COPY_BUTTON_LABEL.get(),
            INFO_ERROR_BROWSER_COPY_BUTTON_TOOLTIP.get());
    copyButton.addActionListener(new ActionListener()
    {
      public void actionPerformed(ActionEvent ev)
      {
        StringSelection s = new StringSelection(url);
        getToolkit().getSystemClipboard().setContents(s, s);
      }
    });
    gbc.insets.left = UIFactory.LEFT_INSET_COPY_BUTTON;
    gbc.insets.right = pInsets.right;
    gbc.fill = GridBagConstraints.NONE;
    p1.add(copyButton, gbc);
    gbc.weighty = 1.0;
    gbc.fill = GridBagConstraints.VERTICAL;
    p1.add(Box.createVerticalGlue(), gbc);

    JPanel p2 = new JPanel(new GridBagLayout());
    p2.setOpaque(false);
    gbc.fill = GridBagConstraints.HORIZONTAL;
    gbc.weightx = 1.0;
    gbc.insets = UIFactory.getEmptyInsets();
    gbc.gridwidth = GridBagConstraints.RELATIVE;
    p2.add(Box.createHorizontalGlue(), gbc);
    JButton closeButton =
        UIFactory.makeJButton(INFO_CLOSE_BUTTON_LABEL.get(),
            INFO_ERROR_BROWSER_CLOSE_BUTTON_TOOLTIP.get());
    gbc.fill = GridBagConstraints.NONE;
    gbc.weightx = 0.0;
    gbc.gridwidth = GridBagConstraints.REMAINDER;
    p2.add(closeButton, gbc);
    closeButton.addActionListener(new ActionListener()
    {
      public void actionPerformed(ActionEvent ev)
      {
        dispose();
      }
    });

    JPanel p = new JPanel(new GridBagLayout());
    p.setBackground(UIFactory.DEFAULT_BACKGROUND);
    gbc.insets = UIFactory.getEmptyInsets();
    gbc.fill = GridBagConstraints.BOTH;
    gbc.gridwidth = GridBagConstraints.REMAINDER;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    p.add(p1, gbc);
    gbc.weighty = 0.0;
    gbc.insets = UIFactory.getButtonsPanelInsets();
    p.add(p2, gbc);
    getRootPane().setDefaultButton(copyButton);
    return p;
  }

  /**
   * Method written for testing purposes.
   * @param args the arguments to be passed to the test program.
   */
  public static void main(String[] args)
  {
    try
    {
      // UIFactory.initialize();
      WebBrowserErrorDialog dlg =
          new WebBrowserErrorDialog(new JFrame(),
              new WebBrowserException("http://opendj.org",
                      LocalizableMessage.raw("toto"), null));
      dlg.packAndShow();
    } catch (Exception ex)
    {
      ex.printStackTrace();
    }
  }
}
