/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at legal-notices/CDDLv1_0.txt
 * or http://forgerock.org/license/CDDLv1.0.html.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at legal-notices/CDDLv1_0.txt.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2009-2010 Sun Microsystems, Inc.
 *      Portions Copyright 2014-2015 ForgeRock AS
 */
package org.opends.guitools.controlpanel.ui;

import static org.opends.messages.AdminToolMessages.*;

import java.awt.Component;
import java.awt.GridBagConstraints;
import java.util.LinkedHashSet;

import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JTextField;

import org.opends.guitools.controlpanel.event.ConfigurationChangeEvent;
import org.opends.guitools.controlpanel.util.Utilities;
import org.forgerock.i18n.LocalizableMessage;

/**
 * The panel that displays the refresh options of the control panel.  Basically
 * it allows to set the refreshing period used by the control panel.
 *
 */
public class RefreshOptionsPanel extends StatusGenericPanel
{
  private static final long serialVersionUID = 641533296295459469L;
  private JTextField period;
  private JLabel lPeriod;

  private boolean isCanceled = true;

  private int MAX_VALUE = 5000;

  /**
   * Default constructor.
   *
   */
  public RefreshOptionsPanel()
  {
    super();
    createLayout();
  }

  /** {@inheritDoc} */
  public LocalizableMessage getTitle()
  {
    return INFO_CTRL_PANEL_REFRESH_PANEL_TITLE.get();
  }

  /**
   * Creates the layout of the panel (but the contents are not populated here).
   */
  private void createLayout()
  {
    GridBagConstraints gbc = new GridBagConstraints();
    gbc.anchor = GridBagConstraints.WEST;
    gbc.weightx = 0.0;
    gbc.gridx = 0;
    gbc.gridy = 0;
    gbc.gridwidth = 2;
    gbc.weightx = 1.0;
    gbc.fill = GridBagConstraints.BOTH;

    String text = INFO_CTRL_PANEL_REFRESH_OPTIONS_PANEL_TEXT.get().toString();

    JEditorPane pane = Utilities.makeHtmlPane(text,
        ColorAndFontConstants.defaultFont);

    Utilities.updatePreferredSize(pane, 60, text,
        ColorAndFontConstants.defaultFont, false);
    gbc.weighty = 0.0;
    add(pane, gbc);

    gbc.gridy = 1;
    gbc.gridwidth = 1;
    gbc.weightx = 0.0;
    gbc.weighty = 0.0;
    lPeriod =Utilities.createPrimaryLabel(
        INFO_CTRL_PANEL_REFRESH_OPTIONS_LABEL.get());
    gbc.insets.top = 10;
    add(lPeriod, gbc);
    period = Utilities.createShortTextField();
    gbc.insets.left = 10;
    gbc.gridx = 1;
    gbc.weightx = 1.0;
    add(period, gbc);

    gbc.gridwidth = 2;
    addBottomGlue(gbc);
  }

  /** {@inheritDoc} */
  public GenericDialog.ButtonType getButtonType()
  {
    return GenericDialog.ButtonType.OK_CANCEL;
  }

  /** {@inheritDoc} */
  public Component getPreferredFocusComponent()
  {
    return period;
  }

  /** {@inheritDoc} */
  public void configurationChanged(ConfigurationChangeEvent ev)
  {
  }

  /** {@inheritDoc} */
  public void okClicked()
  {
    isCanceled = true;

    setPrimaryValid(lPeriod);
    LinkedHashSet<LocalizableMessage> errors = new LinkedHashSet<>();
    long t = -1;
    try
    {
      t = Long.parseLong(period.getText());
    }
    catch (Throwable th)
    {
    }
    if (t <= 0 || t > MAX_VALUE)
    {
      errors.add(INFO_CTRL_PANEL_INVALID_PERIOD_VALUE.get(MAX_VALUE));
    }

    if (!errors.isEmpty())
    {
      displayErrorDialog(errors);
    }
    else
    {
      isCanceled = false;
      Utilities.getParentDialog(this).setVisible(false);
    }
  }

  /**
   * Returns whether this dialog has been cancelled or not.
   * @return whether this dialog has been cancelled or not.
   */
  public boolean isCanceled()
  {
    return isCanceled;
  }

  /** {@inheritDoc} */
  public void toBeDisplayed(boolean visible)
  {
    if (visible)
    {
      isCanceled = true;
      long timeInSeconds = getInfo().getPoolingPeriod() / 1000;
      period.setText(String.valueOf(timeInSeconds));
    }
  }

  /**
   * Returns the time specified by the user in milliseconds.
   * @return the time specified by the user in milliseconds.
   */
  public long getPoolingPeriod()
  {
    long t = -1;
    try
    {
      t = 1000 * Long.parseLong(period.getText());
    }
    catch (Throwable th)
    {
    }
    return t;
  }
}
