/*
 * Decompiled with CFR 0.152.
 */
package org.forgerock.audit;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import org.forgerock.audit.AuditException;
import org.forgerock.audit.AuditService;
import org.forgerock.audit.AuditServiceConfiguration;
import org.forgerock.audit.AuditingContext;
import org.forgerock.audit.events.EventTopicsMetaData;
import org.forgerock.audit.events.handlers.AuditEventHandler;
import org.forgerock.audit.filter.Filter;
import org.forgerock.audit.filter.FilterChainBuilder;
import org.forgerock.audit.util.ResourceExceptionsUtil;
import org.forgerock.json.JsonValue;
import org.forgerock.json.resource.ActionRequest;
import org.forgerock.json.resource.ActionResponse;
import org.forgerock.json.resource.BadRequestException;
import org.forgerock.json.resource.CreateRequest;
import org.forgerock.json.resource.DeleteRequest;
import org.forgerock.json.resource.NotSupportedException;
import org.forgerock.json.resource.PatchRequest;
import org.forgerock.json.resource.QueryRequest;
import org.forgerock.json.resource.QueryResourceHandler;
import org.forgerock.json.resource.QueryResponse;
import org.forgerock.json.resource.ReadRequest;
import org.forgerock.json.resource.Request;
import org.forgerock.json.resource.ResourceException;
import org.forgerock.json.resource.ResourcePath;
import org.forgerock.json.resource.ResourceResponse;
import org.forgerock.json.resource.Responses;
import org.forgerock.json.resource.ServiceUnavailableException;
import org.forgerock.json.resource.UpdateRequest;
import org.forgerock.services.context.Context;
import org.forgerock.util.generator.IdGenerator;
import org.forgerock.util.promise.Promise;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class AuditServiceImpl
implements AuditService {
    private static final Logger logger = LoggerFactory.getLogger(AuditServiceImpl.class);
    private final AuditServiceConfiguration config;
    private final Map<String, AuditEventHandler> auditEventHandlersByName;
    private final Map<String, Set<AuditEventHandler>> auditEventHandlersByTopic;
    private final EventTopicsMetaData eventTopicsMetaData;
    private final AuditEventHandler queryHandler;
    private volatile LifecycleState lifecycleState = LifecycleState.STARTING;
    private final Filter filters;

    public AuditServiceImpl(AuditServiceConfiguration configuration, EventTopicsMetaData eventTopicsMetaData, Set<AuditEventHandler> auditEventHandlers) {
        this.config = new AuditServiceConfiguration(configuration);
        this.eventTopicsMetaData = eventTopicsMetaData;
        this.auditEventHandlersByName = this.getAuditEventHandlersByName(auditEventHandlers);
        this.auditEventHandlersByTopic = this.getAuditEventHandlersByTopic(auditEventHandlers, eventTopicsMetaData);
        String queryHandlerName = configuration.getHandlerForQueries();
        this.queryHandler = queryHandlerName != null && this.auditEventHandlersByName.containsKey(queryHandlerName) ? this.auditEventHandlersByName.get(queryHandlerName) : new NullQueryHandler(this.config.getHandlerForQueries());
        this.filters = new FilterChainBuilder().withAuditTopics(eventTopicsMetaData.getTopics()).withPolicies(configuration.getFilterPolicies()).build();
    }

    private Map<String, AuditEventHandler> getAuditEventHandlersByName(Set<AuditEventHandler> handlers) {
        HashMap<String, AuditEventHandler> handlersByName = new HashMap<String, AuditEventHandler>(handlers.size());
        for (AuditEventHandler handler : handlers) {
            handlersByName.put(handler.getName(), handler);
        }
        return handlersByName;
    }

    private Map<String, Set<AuditEventHandler>> getAuditEventHandlersByTopic(Set<AuditEventHandler> handlers, EventTopicsMetaData eventTopicsMetaData) {
        HashMap<String, Set<AuditEventHandler>> handlersByTopic = new HashMap<String, Set<AuditEventHandler>>();
        for (String topic : eventTopicsMetaData.getTopics()) {
            handlersByTopic.put(topic, new LinkedHashSet());
        }
        for (AuditEventHandler handler : handlers) {
            if (!handler.isEnabled()) continue;
            for (String topic : handler.getHandledTopics()) {
                ((Set)handlersByTopic.get(topic)).add(handler);
            }
        }
        return handlersByTopic;
    }

    @Override
    public Promise<ResourceResponse, ResourceException> handleRead(Context context, ReadRequest request) {
        try {
            logger.debug("Audit read called for {}", (Object)request.getResourcePath());
            this.checkLifecycleStateIsRunning();
            if (request.getResourcePathObject().size() != 2) {
                return new BadRequestException("Invalid resource path object specified.").asPromise();
            }
            String id = request.getResourcePathObject().tail(1).toString();
            String topic = this.establishTopic(request.getResourcePathObject(), true);
            return this.queryHandler.readEvent(context, topic, id);
        }
        catch (Exception e) {
            return ResourceExceptionsUtil.adapt(e).asPromise();
        }
    }

    @Override
    public Promise<ResourceResponse, ResourceException> handleCreate(Context context, CreateRequest request) {
        try {
            logger.trace("Audit create called for {}", (Object)request.getResourcePath());
            this.checkLifecycleStateIsRunning();
            if (context.containsContext(AuditingContext.class)) {
                return this.newUnhandledEventResponse().asPromise();
            }
            String topic = this.establishTopic(request.getResourcePathObject(), true);
            this.rejectIfMissingTransactionIdOrTimestamp(request);
            this.establishAuditEventId(request);
            this.filters.doFilter(topic, request.getContent());
            Collection<AuditEventHandler> auditEventHandlersForEvent = this.getAuditEventHandlersForEvent(topic);
            return this.publishEventToHandlers(context, request.getContent(), topic, auditEventHandlersForEvent);
        }
        catch (Exception e) {
            logger.warn(e.getMessage());
            return ResourceExceptionsUtil.adapt(e).asPromise();
        }
    }

    private ResourceResponse newUnhandledEventResponse() {
        return Responses.newResourceResponse(null, null, (JsonValue)JsonValue.json((Object)JsonValue.object((Map.Entry[])new Map.Entry[0])));
    }

    private void rejectIfMissingTransactionIdOrTimestamp(CreateRequest request) throws BadRequestException {
        if (!request.getContent().isDefined("transactionId") || !request.getContent().isDefined("timestamp")) {
            throw new BadRequestException("The request requires a transactionId and a timestamp");
        }
    }

    private void establishAuditEventId(CreateRequest request) {
        String newResourceId = request.getNewResourceId();
        String auditEventId = newResourceId == null || newResourceId.isEmpty() ? IdGenerator.DEFAULT.generate() : newResourceId;
        request.getContent().put("_id", (Object)auditEventId);
        logger.debug("Audit create id {}", (Object)auditEventId);
    }

    private String establishTopic(ResourcePath path, boolean isTopicRequired) throws ResourceException {
        String topic = path.head(1).toString();
        if (isTopicRequired && topic == null) {
            throw new BadRequestException("Audit service called without specifying event topic in the identifier");
        }
        if (topic != null && !this.eventTopicsMetaData.containsTopic(topic)) {
            throw new NotSupportedException("Audit service called with unknown event topic " + topic);
        }
        return topic;
    }

    private Promise<ResourceResponse, ResourceException> publishEventToHandlers(Context context, JsonValue event, String topic, Collection<AuditEventHandler> auditEventHandlersForEvent) {
        Promise promise = this.newUnhandledEventResponse().asPromise();
        if (auditEventHandlersForEvent.isEmpty()) {
            logger.debug("No handler found for the event of topic {}", (Object)topic);
            return promise;
        }
        logger.debug("Cascading the event of topic {} to the handlers : {}", (Object)topic, auditEventHandlersForEvent);
        for (AuditEventHandler auditEventHandler : auditEventHandlersForEvent) {
            Promise handlerResult;
            try {
                handlerResult = auditEventHandler.publishEvent(context, topic, event);
            }
            catch (Exception ex) {
                logger.warn(ex.getMessage());
                handlerResult = ResourceExceptionsUtil.adapt(ex).asPromise();
            }
            if (auditEventHandler != this.queryHandler) continue;
            promise = handlerResult;
        }
        return promise;
    }

    @Override
    public Promise<ResourceResponse, ResourceException> handleUpdate(Context context, UpdateRequest request) {
        return ResourceExceptionsUtil.notSupported((Request)request).asPromise();
    }

    @Override
    public Promise<ResourceResponse, ResourceException> handleDelete(Context context, DeleteRequest request) {
        return ResourceExceptionsUtil.notSupported((Request)request).asPromise();
    }

    @Override
    public Promise<ResourceResponse, ResourceException> handlePatch(Context context, PatchRequest request) {
        return ResourceExceptionsUtil.notSupported((Request)request).asPromise();
    }

    @Override
    public Promise<QueryResponse, ResourceException> handleQuery(Context context, QueryRequest request, QueryResourceHandler handler) {
        try {
            logger.debug("Audit query called for {}", (Object)request.getResourcePath());
            this.checkLifecycleStateIsRunning();
            String topic = this.establishTopic(request.getResourcePathObject(), true);
            return this.queryHandler.queryEvents(context, topic, request, handler);
        }
        catch (Exception e) {
            return ResourceExceptionsUtil.adapt(e).asPromise();
        }
    }

    @Override
    public Promise<ActionResponse, ResourceException> handleAction(Context context, ActionRequest request) {
        try {
            String handlerName = request.getAdditionalParameter("handler");
            String topic = this.establishTopic(request.getResourcePathObject(), false);
            if (handlerName == null) {
                return new BadRequestException(String.format("Unable to handle action: %s", request.getAction())).asPromise();
            }
            AuditEventHandler handler = this.auditEventHandlersByName.get(handlerName);
            if (handler == null) {
                return new BadRequestException(String.format("Action references an unknown handler name: %s", handlerName)).asPromise();
            }
            return handler.handleAction(context, topic, request);
        }
        catch (Exception e) {
            return ResourceExceptionsUtil.adapt(e).asPromise();
        }
    }

    private Collection<AuditEventHandler> getAuditEventHandlersForEvent(String auditEvent) {
        if (this.auditEventHandlersByTopic.containsKey(auditEvent)) {
            return this.auditEventHandlersByTopic.get(auditEvent);
        }
        return Collections.emptyList();
    }

    @Override
    public AuditServiceConfiguration getConfig() throws ServiceUnavailableException {
        this.checkLifecycleStateIsRunning();
        return new AuditServiceConfiguration(this.config);
    }

    @Override
    public AuditEventHandler getRegisteredHandler(String handlerName) throws ServiceUnavailableException {
        this.checkLifecycleStateIsRunning();
        return this.auditEventHandlersByName.get(handlerName);
    }

    @Override
    public boolean isAuditing(String topic) throws ServiceUnavailableException {
        this.checkLifecycleStateIsRunning();
        return !this.getAuditEventHandlersForEvent(topic).isEmpty();
    }

    @Override
    public Set<String> getKnownTopics() throws ServiceUnavailableException {
        this.checkLifecycleStateIsRunning();
        return this.eventTopicsMetaData.getTopics();
    }

    @Override
    public void startup() throws ServiceUnavailableException {
        switch (this.lifecycleState) {
            case STARTING: {
                for (Map.Entry<String, AuditEventHandler> entry : this.auditEventHandlersByName.entrySet()) {
                    String handlerName = entry.getKey();
                    AuditEventHandler handler = entry.getValue();
                    try {
                        handler.startup();
                    }
                    catch (ResourceException e) {
                        logger.warn("Unable to startup handler " + handlerName, (Throwable)e);
                    }
                }
                this.lifecycleState = LifecycleState.RUNNING;
                break;
            }
            case RUNNING: {
                break;
            }
            case SHUTDOWN: {
                throw new ServiceUnavailableException("AuditService cannot be restarted after shutdown");
            }
            default: {
                throw new IllegalStateException("AuditService is in an unknown state");
            }
        }
    }

    @Override
    public void shutdown() {
        switch (this.lifecycleState) {
            case STARTING: {
                this.lifecycleState = LifecycleState.SHUTDOWN;
                break;
            }
            case RUNNING: {
                for (Map.Entry<String, AuditEventHandler> entry : this.auditEventHandlersByName.entrySet()) {
                    String handlerName = entry.getKey();
                    AuditEventHandler handler = entry.getValue();
                    try {
                        handler.shutdown();
                    }
                    catch (ResourceException e) {
                        logger.warn("Unable to shutdown handler " + handlerName, (Throwable)e);
                    }
                }
                this.lifecycleState = LifecycleState.SHUTDOWN;
                break;
            }
            case SHUTDOWN: {
                break;
            }
            default: {
                throw new IllegalStateException("AuditService is in an unknown state");
            }
        }
    }

    @Override
    public boolean isRunning() {
        return this.lifecycleState == LifecycleState.RUNNING;
    }

    private void checkLifecycleStateIsRunning() throws ServiceUnavailableException {
        if (this.lifecycleState != LifecycleState.RUNNING) {
            throw new ServiceUnavailableException("AuditService not running");
        }
    }

    private class NullQueryHandler
    implements AuditEventHandler {
        private final String errorMessage;

        private NullQueryHandler(String handlerForQueries) {
            this.errorMessage = handlerForQueries == null || handlerForQueries.trim().isEmpty() ? "No handler defined for queries." : "The handler defined for queries, '" + handlerForQueries + "', has not been registered to the audit service.";
        }

        @Override
        public void startup() throws ResourceException {
            throw new UnsupportedOperationException("Unsupported.");
        }

        @Override
        public void shutdown() throws ResourceException {
            throw new UnsupportedOperationException("Unsupported.");
        }

        @Override
        public String getName() {
            throw new UnsupportedOperationException("Unsupported.");
        }

        @Override
        public Set<String> getHandledTopics() {
            throw new UnsupportedOperationException("Unsupported.");
        }

        @Override
        public Promise<ResourceResponse, ResourceException> publishEvent(Context context, String topic, JsonValue event) {
            throw new UnsupportedOperationException("Unsupported.");
        }

        @Override
        public Promise<ResourceResponse, ResourceException> readEvent(Context context, String topic, String resourceId) {
            return ResourceExceptionsUtil.adapt(new AuditException(this.errorMessage)).asPromise();
        }

        @Override
        public Promise<QueryResponse, ResourceException> queryEvents(Context context, String topic, QueryRequest query, QueryResourceHandler handler) {
            return ResourceExceptionsUtil.adapt(new AuditException(this.errorMessage)).asPromise();
        }

        @Override
        public boolean isEnabled() {
            return true;
        }

        @Override
        public Promise<ActionResponse, ResourceException> handleAction(Context context, String topic, ActionRequest request) {
            throw new UnsupportedOperationException("Unsupported.");
        }
    }

    private static enum LifecycleState {
        STARTING,
        RUNNING,
        SHUTDOWN;

    }
}

