/*
 * The contents of this file are subject to the terms of the Common Development and
 * Distribution License (the License). You may not use this file except in compliance with the
 * License.
 *
 * You can obtain a copy of the License at legal/CDDLv1.0.txt. See the License for the
 * specific language governing permission and limitations under the License.
 *
 * When distributing Covered Software, include this CDDL Header Notice in each file and include
 * the License file at legal/CDDLv1.0.txt. If applicable, add the following below the CDDL
 * Header, with the fields enclosed by brackets [] replaced by your own identifying
 * information: "Portions Copyright [year] [name of copyright owner]".
 *
 * Copyright 2008 Sun Microsystems, Inc.
 * Portions Copyright 2014-2015 ForgeRock AS.
 */

package org.opends.quicksetup;
import org.forgerock.i18n.LocalizableMessage;

import org.opends.quicksetup.event.ProgressUpdateListener;
import org.opends.quicksetup.event.ProgressUpdateEvent;
import java.util.HashSet;

/**
 * Delegate class for handling progress notification listeners and events.
 */
public class ProgressUpdateListenerDelegate {

  private HashSet<ProgressUpdateListener> listeners = new HashSet<>();

  /**
   * Creates a parameterized instance.
   */
  public ProgressUpdateListenerDelegate() {
  }

  /**
   * Adds a ProgressUpdateListener that will be notified of updates in
   * the install progress.
   *
   * @param l the ProgressUpdateListener to be added.
   */
  public void addProgressUpdateListener(ProgressUpdateListener l) {
    listeners.add(l);
  }

  /**
   * Removes a ProgressUpdateListener.
   *
   * @param l the ProgressUpdateListener to be removed.
   */
  public void removeProgressUpdateListener(ProgressUpdateListener l) {
    listeners.remove(l);
  }

  /**
   * This method notifies the ProgressUpdateListeners that there was an
   * update in the installation progress.
   *
   * @param current             progress step
   * @param ratio               the integer that specifies which percentage of
 *                            the whole installation has been completed.
   * @param currentPhaseSummary the localized summary message for the
*                            current installation progress in formatted form.
   * @param newLogDetail        the new log messages that we have for the
   */
  public void notifyListeners(ProgressStep current, Integer ratio,
                              LocalizableMessage currentPhaseSummary,
                              LocalizableMessage newLogDetail) {
    ProgressUpdateEvent ev =
            new ProgressUpdateEvent(current, ratio,
                    currentPhaseSummary, newLogDetail);
    for (ProgressUpdateListener l : listeners) {
      l.progressUpdate(ev);
    }
  }

  /**
   * Notify listeners about a change in log detail.
   * @param msg log detail
   */
  protected void notifyListeners(LocalizableMessage msg) {
    notifyListeners(null, null, null, msg);
  }

}
