/*
 * Decompiled with CFR 0.152.
 */
package org.forgerock.opendj.security;

import java.security.Key;
import javax.crypto.spec.SecretKeySpec;
import org.assertj.core.api.Assertions;
import org.forgerock.opendj.ldap.ByteSequence;
import org.forgerock.opendj.ldap.ByteString;
import org.forgerock.opendj.ldap.ByteStringBuilder;
import org.forgerock.opendj.ldap.SdkTestCase;
import org.forgerock.opendj.security.ExternalKeyWrappingStrategy;
import org.forgerock.opendj.security.KeyProtector;
import org.forgerock.opendj.security.KeyStoreParameters;
import org.forgerock.opendj.security.KeyStoreTestUtils;
import org.forgerock.opendj.security.LocalizedKeyStoreException;
import org.forgerock.opendj.security.OpenDJProvider;
import org.forgerock.util.Options;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

public class KeyProtectorTest
extends SdkTestCase {
    private static final ByteString PLAIN_KEY_NEEDS_PADDING = ByteString.valueOfUtf8((CharSequence)"123456781234");
    private static final ByteString PLAIN_KEY = ByteString.valueOfUtf8((CharSequence)"1234567812345678");
    private static final char[] KEYSTORE_PASSWORD = "keystore password".toCharArray();
    private static final char[] KEY_PASSWORD = "key password".toCharArray();
    private static final char[] BAD_PASSWORD = "bad password".toCharArray();
    private static final ExternalKeyWrappingStrategy TEST_STRATEGY = new ExternalKeyWrappingStrategy(){

        public ByteSequence wrapKey(ByteSequence unwrappedKey) throws LocalizedKeyStoreException {
            return ByteString.valueOfUtf8((CharSequence)unwrappedKey.toBase64String());
        }

        public ByteSequence unwrapKey(ByteSequence wrappedKey) throws LocalizedKeyStoreException {
            return ByteString.valueOfBase64((String)wrappedKey.toString());
        }
    };

    @BeforeClass
    public void sanityCheckTestFramework() throws Exception {
        Assertions.assertThat((boolean)this.encodedKeyIsEncrypted(PLAIN_KEY)).isFalse();
        Assertions.assertThat((boolean)this.encodedKeyIsEncrypted(new ByteStringBuilder().appendBytes((ByteSequence)PLAIN_KEY).appendUtf8("tail").toByteString())).isFalse();
        Assertions.assertThat((boolean)this.encodedKeyIsEncrypted(new ByteStringBuilder().appendUtf8("head").appendBytes((ByteSequence)PLAIN_KEY).toByteString())).isFalse();
        Assertions.assertThat((boolean)this.encodedKeyIsEncrypted(new ByteStringBuilder().appendUtf8("head").appendBytes((ByteSequence)PLAIN_KEY).appendUtf8("tail").toByteString())).isFalse();
        Assertions.assertThat((boolean)this.encodedKeyIsEncrypted(ByteString.valueOfUtf8((CharSequence)"different"))).isTrue();
    }

    @Test
    public void shouldEncodeAndDecodeWithNoProtection() throws Exception {
        KeyProtector kp = new KeyProtector(Options.defaultOptions());
        ByteString encodedKey = kp.encodeKey(KeyProtectorTest.secretKey(PLAIN_KEY), null);
        Assertions.assertThat((Comparable)encodedKey).isNotEqualTo((Object)PLAIN_KEY);
        Assertions.assertThat((boolean)this.encodedKeyIsEncrypted(encodedKey)).isFalse();
        Assertions.assertThat((byte[])kp.decodeSecretKey((ByteSequence)encodedKey, "RAW", null).getEncoded()).isEqualTo((Object)PLAIN_KEY.toByteArray());
    }

    @Test
    public void shouldEncodeAndDecodeWithIndividualPassword() throws Exception {
        this.shouldEncodeAndDecodeAndBeEncrypted(new KeyProtector(Options.defaultOptions()), KEY_PASSWORD);
    }

    @Test
    public void shouldEncodeAndDecodeWithKeyStorePasswordOnly() throws Exception {
        this.shouldEncodeAndDecodeAndBeEncrypted(KeyProtectorTest.forPasswordProtectedKeyStore(KEYSTORE_PASSWORD), null);
    }

    private static KeyProtector forPasswordProtectedKeyStore(char[] keystorePassword) {
        return new KeyProtector(Options.defaultOptions().set(KeyStoreParameters.GLOBAL_PASSWORD, (Object)OpenDJProvider.newClearTextPasswordFactory((char[])keystorePassword)));
    }

    @Test
    public void shouldEncodeAndDecodeWithKeyStorePasswordAndIndividualPassword() throws Exception {
        this.shouldEncodeAndDecodeAndBeEncrypted(KeyProtectorTest.forPasswordProtectedKeyStore(KEYSTORE_PASSWORD), KEY_PASSWORD);
    }

    private static KeyProtector forExternallyProtectedKeyStore() {
        return new KeyProtector(Options.defaultOptions().set(KeyStoreParameters.EXTERNAL_KEY_WRAPPING_STRATEGY, (Object)TEST_STRATEGY));
    }

    @Test
    public void shouldEncodeAndDecodeWithExternalMechanismOnly() throws Exception {
        this.shouldEncodeAndDecodeAndBeEncrypted(KeyProtectorTest.forExternallyProtectedKeyStore(), null);
    }

    @Test
    public void shouldEncodeAndDecodeWithExternalMechanismAndIndividualPassword() throws Exception {
        this.shouldEncodeAndDecodeAndBeEncrypted(KeyProtectorTest.forExternallyProtectedKeyStore(), KEY_PASSWORD);
    }

    @Test(expectedExceptions={LocalizedKeyStoreException.class})
    public void shouldFailWhenDecodeWithWrongIndividualPassword() throws Exception {
        KeyProtector kpEncode = KeyProtectorTest.forPasswordProtectedKeyStore(KEYSTORE_PASSWORD);
        ByteString encodedKey = kpEncode.encodeKey(KeyProtectorTest.secretKey(PLAIN_KEY), KEY_PASSWORD);
        KeyProtector kpDecode = KeyProtectorTest.forPasswordProtectedKeyStore(KEYSTORE_PASSWORD);
        kpDecode.decodeSecretKey((ByteSequence)encodedKey, "RAW", BAD_PASSWORD);
    }

    @Test(expectedExceptions={LocalizedKeyStoreException.class})
    public void shouldFailWhenDecodeWithWrongKeyStorePassword() throws Exception {
        KeyProtector kpEncode = KeyProtectorTest.forPasswordProtectedKeyStore(KEYSTORE_PASSWORD);
        ByteString encodedKey = kpEncode.encodeKey(KeyProtectorTest.secretKey(PLAIN_KEY), KEY_PASSWORD);
        KeyProtector kpDecode = KeyProtectorTest.forPasswordProtectedKeyStore(BAD_PASSWORD);
        kpDecode.decodeSecretKey((ByteSequence)encodedKey, "RAW", KEY_PASSWORD);
    }

    @Test(expectedExceptions={LocalizedKeyStoreException.class})
    public void shouldFailWhenDecodeWithMissingKeyStorePassword() throws Exception {
        KeyProtector kpEncode = KeyProtectorTest.forPasswordProtectedKeyStore(KEYSTORE_PASSWORD);
        ByteString encodedKey = kpEncode.encodeKey(KeyProtectorTest.secretKey(PLAIN_KEY), KEY_PASSWORD);
        KeyProtector kpDecode = KeyProtectorTest.forPasswordProtectedKeyStore(null);
        kpDecode.decodeSecretKey((ByteSequence)encodedKey, "RAW", KEY_PASSWORD);
    }

    @Test(expectedExceptions={LocalizedKeyStoreException.class})
    public void shouldFailWhenDecodeWithMissingIndividualPassword() throws Exception {
        KeyProtector kpEncode = KeyProtectorTest.forPasswordProtectedKeyStore(KEYSTORE_PASSWORD);
        ByteString encodedKey = kpEncode.encodeKey(KeyProtectorTest.secretKey(PLAIN_KEY), KEY_PASSWORD);
        KeyProtector kpDecode = KeyProtectorTest.forPasswordProtectedKeyStore(KEYSTORE_PASSWORD);
        kpDecode.decodeSecretKey((ByteSequence)encodedKey, "RAW", null);
    }

    @Test(expectedExceptions={LocalizedKeyStoreException.class})
    public void shouldFailWhenDecodeWithMissingPasswords() throws Exception {
        KeyProtector kpEncode = KeyProtectorTest.forPasswordProtectedKeyStore(KEYSTORE_PASSWORD);
        ByteString encodedKey = kpEncode.encodeKey(KeyProtectorTest.secretKey(PLAIN_KEY), KEY_PASSWORD);
        KeyProtector kpDecode = KeyProtectorTest.forPasswordProtectedKeyStore(null);
        kpDecode.decodeSecretKey((ByteSequence)encodedKey, "RAW", null);
    }

    @Test(expectedExceptions={LocalizedKeyStoreException.class})
    public void shouldFailWhenDecodeWithMalformedEncodedKey() throws Exception {
        KeyProtector kp = KeyProtectorTest.forPasswordProtectedKeyStore(KEYSTORE_PASSWORD);
        kp.decodeSecretKey((ByteSequence)ByteString.valueOfUtf8((CharSequence)"malformed encoded key"), "RAW", KEY_PASSWORD);
    }

    @Test
    public void shouldEncodeAndDecodeKeysThatNeedPadding() throws Exception {
        KeyProtector kp = KeyProtectorTest.forPasswordProtectedKeyStore(KEYSTORE_PASSWORD);
        ByteString encodedKey = kp.encodeKey(KeyProtectorTest.secretKey(PLAIN_KEY_NEEDS_PADDING), KEY_PASSWORD);
        Assertions.assertThat((Comparable)encodedKey).isNotEqualTo((Object)PLAIN_KEY_NEEDS_PADDING);
        byte[] decodedKey = kp.decodeSecretKey((ByteSequence)encodedKey, "RAW", KEY_PASSWORD).getEncoded();
        Assertions.assertThat((byte[])decodedKey).isEqualTo((Object)PLAIN_KEY_NEEDS_PADDING.toByteArray());
    }

    @Test
    public void shouldEncodeAndDecodePrivateKeys() throws Exception {
        KeyProtector kp = KeyProtectorTest.forPasswordProtectedKeyStore(KEYSTORE_PASSWORD);
        ByteString encodedKey = kp.encodeKey((Key)KeyStoreTestUtils.PRIVATE_KEY, KEY_PASSWORD);
        Assertions.assertThat((Comparable)encodedKey).isNotEqualTo((Object)ByteString.wrap((byte[])KeyStoreTestUtils.PRIVATE_KEY.getEncoded()));
        byte[] decodedKey = kp.decodePrivateKey((ByteSequence)encodedKey, "RSA", KEY_PASSWORD).getEncoded();
        Assertions.assertThat((byte[])decodedKey).isEqualTo((Object)KeyStoreTestUtils.PRIVATE_KEY.getEncoded());
    }

    private void shouldEncodeAndDecodeAndBeEncrypted(KeyProtector kp, char[] password) throws Exception {
        ByteString encodedKey = kp.encodeKey(KeyProtectorTest.secretKey(PLAIN_KEY), password);
        Assertions.assertThat((Comparable)encodedKey).isNotEqualTo((Object)PLAIN_KEY);
        Assertions.assertThat((boolean)this.encodedKeyIsEncrypted(encodedKey)).isTrue();
        Assertions.assertThat((byte[])kp.decodeSecretKey((ByteSequence)encodedKey, "RAW", password).getEncoded()).isEqualTo((Object)PLAIN_KEY.toByteArray());
    }

    private boolean encodedKeyIsEncrypted(ByteString encodedKey) {
        int end = encodedKey.length() - PLAIN_KEY.length();
        for (int i = 0; i <= end; ++i) {
            ByteString subSequence = encodedKey.subSequence(i, i + PLAIN_KEY.length());
            if (!subSequence.equals((Object)PLAIN_KEY)) continue;
            return false;
        }
        return true;
    }

    private static Key secretKey(ByteString rawKey) {
        return new SecretKeySpec(rawKey.toByteArray(), "RAW");
    }
}

