/*
 * Decompiled with CFR 0.152.
 */
package org.forgerock.opendj.ldif;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.NoSuchElementException;
import org.fest.assertions.Assertions;
import org.forgerock.i18n.LocalizableMessage;
import org.forgerock.i18n.LocalizedIllegalArgumentException;
import org.forgerock.opendj.ldap.AttributeDescription;
import org.forgerock.opendj.ldap.DN;
import org.forgerock.opendj.ldap.DecodeException;
import org.forgerock.opendj.ldap.Entry;
import org.forgerock.opendj.ldap.Filter;
import org.forgerock.opendj.ldap.LinkedHashMapEntry;
import org.forgerock.opendj.ldap.Matcher;
import org.forgerock.opendj.ldap.TestCaseUtils;
import org.forgerock.opendj.ldap.schema.Schema;
import org.forgerock.opendj.ldap.schema.SchemaBuilder;
import org.forgerock.opendj.ldap.schema.SchemaValidationPolicy;
import org.forgerock.opendj.ldif.AbstractLDIFTestCase;
import org.forgerock.opendj.ldif.LDIFEntryReader;
import org.forgerock.opendj.ldif.RejectedLDIFListener;
import org.mockito.Matchers;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.testng.Assert;
import org.testng.annotations.Test;

public final class LDIFEntryReaderTestCase
extends AbstractLDIFTestCase {
    public final int nbStandardEntryAttributes = new LinkedHashMapEntry(this.getStandardEntry()).getAttributeCount();

    public final String[] getStandardEntry() {
        return new String[]{"dn: uid=user.0,ou=People,dc=example,dc=com", "objectClass: person", "objectClass: inetorgperson", "objectClass: organizationalperson", "objectClass: top", "postalAddress: Aaccf Amar$01251 Chestnut Street$Panama City, DE  50369", "postalCode: 50369", "uid: user.0", "description: This is the description for Aaccf Amar.", "userPassword: {SSHA}hpbT8dLi8xgYy2kl4aP6QKGzsFdhESWpPmDTEw==", "employeeNumber: 0", "initials: ASA", "givenName: Aaccf", "pager: +1 779 041 6341", "mobile: +1 010 154 3228", "cn: Aaccf Amar", "telephoneNumber: +1 685 622 6202", "sn: Amar", "street: 01251 Chestnut Street", "homePhone: +1 225 216 5900", "mail: user.0@maildomain.net", "l: Panama City", "st: DE", "pwdChangedTime: 20120903142126.219Z", "entryDN: uid=user.0,ou=people,dc=example,dc=org", "entryUUID: ad55a34a-763f-358f-93f9-da86f9ecd9e4", "modifyTimestamp: 20120903142126Z", "modifiersName: cn=Internal Client,cn=Root DNs,cn=config"};
    }

    @Test(expectedExceptions={NoSuchElementException.class})
    public void testSetExcludeBranchWithNoMatch() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setExcludeBranch(DN.valueOf((String)"dc=example,dc=com"));
        try {
            reader.readEntry();
        }
        finally {
            reader.close();
        }
    }

    @Test
    public void testSetExcludeBranchWithMatch() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setExcludeBranch(DN.valueOf((String)"dc=example,dc=org"));
        Entry entry = reader.readEntry();
        reader.close();
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(this.nbStandardEntryAttributes);
    }

    @Test(expectedExceptions={NullPointerException.class})
    public void testSetExcludeBranchDoesntAllowNull() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setExcludeBranch(null);
        reader.close();
    }

    @Test
    public void testSetExcludeAllUserAttributesFalse() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setExcludeAllUserAttributes(false);
        Entry entry = reader.readEntry();
        reader.close();
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(this.nbStandardEntryAttributes);
        Assertions.assertThat((Collection)entry.getAttribute("entryDN")).isNotNull();
        Assertions.assertThat((Collection)entry.getAttribute("description")).isNotNull();
    }

    @Test
    public void testSetExcludeAllUserAttributesTrue() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setExcludeAllUserAttributes(true);
        Entry entry = reader.readEntry();
        reader.close();
        Assertions.assertThat((Collection)entry.getAttribute("dn")).isNull();
        Assertions.assertThat((Collection)entry.getAttribute("sn")).isNull();
        Assertions.assertThat((Collection)entry.getAttribute("uid")).isNull();
        Assertions.assertThat((Collection)entry.getAttribute("description")).isNull();
        Assertions.assertThat((Collection)entry.getAttribute("entryDN")).isNotEmpty();
        Assertions.assertThat((Collection)entry.getAttribute("entryUUID")).isNotEmpty();
        Assertions.assertThat((Collection)entry.getAttribute("modifyTimestamp")).isNotNull();
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(4);
    }

    @Test
    public void testSetExcludeAttributeWithNoMatch() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setExcludeAttribute(AttributeDescription.valueOf((String)"vip"));
        Entry entry = reader.readEntry();
        reader.close();
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(this.nbStandardEntryAttributes);
        Assertions.assertThat((Collection)entry.getAttribute("vip")).isNull();
    }

    @Test
    public void testSetExcludeAttributeWithMatch() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setExcludeAttribute(AttributeDescription.valueOf((String)"cn"));
        reader.setExcludeAttribute(AttributeDescription.valueOf((String)"cn"));
        reader.setExcludeAttribute(AttributeDescription.valueOf((String)"sn"));
        reader.setExcludeAttribute(AttributeDescription.valueOf((String)"entryDN"));
        Entry entry = reader.readEntry();
        reader.close();
        Assertions.assertThat((Collection)entry.getAttribute("entryDN")).isNull();
        Assertions.assertThat((Collection)entry.getAttribute("sn")).isNull();
        Assertions.assertThat((Collection)entry.getAttribute("cn")).isNull();
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(this.nbStandardEntryAttributes - 3);
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
    }

    @Test(expectedExceptions={NullPointerException.class})
    public void testSetExcludeAttributeDoesntAllowNull() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setExcludeAttribute(null);
        reader.close();
    }

    @Test
    public void testSetExcludeAllOperationalAttributesFalse() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setExcludeAllOperationalAttributes(false);
        Entry entry = reader.readEntry();
        reader.close();
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(this.nbStandardEntryAttributes);
        Assertions.assertThat((Collection)entry.getAttribute("entryDN")).isNotNull();
        Assertions.assertThat((Collection)entry.getAttribute("entryUUID")).isNotNull();
        Assertions.assertThat((Collection)entry.getAttribute("modifyTimestamp")).isNotNull();
    }

    @Test
    public void testSetExcludeAllOperationalAttributesTrue() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setExcludeAllOperationalAttributes(true);
        Entry entry = reader.readEntry();
        reader.close();
        Assertions.assertThat((Collection)entry.getAttribute("entryDN")).isNull();
        Assertions.assertThat((Collection)entry.getAttribute("entryUUID")).isNull();
        Assertions.assertThat((Collection)entry.getAttribute("modifyTimestamp")).isNull();
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
        Assertions.assertThat((int)entry.getAttributeCount()).isLessThan(this.nbStandardEntryAttributes);
    }

    @Test(expectedExceptions={NullPointerException.class})
    public void testsetExcludeFilterDoesntAllowNull() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setExcludeFilter(null);
        reader.close();
    }

    @Test
    public void testSetExcludeFilterWithMatch() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        Filter filter = Filter.equality((String)"objectclass", (Object)"vip");
        Matcher excludeFilter = filter.matcher();
        reader.setExcludeFilter(excludeFilter);
        Entry entry = reader.readEntry();
        reader.close();
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(this.nbStandardEntryAttributes);
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
        Assertions.assertThat((String)entry.getAttribute("objectclass").toString()).isNotEqualTo((Object)"vip");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(expectedExceptions={NoSuchElementException.class})
    public void testSetExcludeFilterWithNoMatch() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        Filter filter = Filter.equality((String)"objectclass", (Object)"person");
        Matcher excludeFilter = filter.matcher();
        reader.setExcludeFilter(excludeFilter);
        try {
            reader.readEntry();
        }
        finally {
            reader.close();
        }
    }

    @Test
    public void testSetIncludeAttributeWithMatch() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setIncludeAttribute(AttributeDescription.valueOf((String)"cn"));
        reader.setIncludeAttribute(AttributeDescription.valueOf((String)"sn"));
        reader.setIncludeAttribute(AttributeDescription.valueOf((String)"sn"));
        Entry entry = reader.readEntry();
        Assertions.assertThat((Object)entry).isNotNull();
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(2);
        Assertions.assertThat((Collection)entry.getAttribute("cn")).isNotNull();
        Assertions.assertThat((Collection)entry.getAttribute("sn")).isNotNull();
        Assertions.assertThat((Collection)entry.getAttribute("description")).isNull();
        reader.close();
    }

    @Test
    public void testSetIncludeAttributeWithNoMatch() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setIncludeAttribute(AttributeDescription.valueOf((String)"manager"));
        Entry entry = reader.readEntry();
        Assertions.assertThat((Object)entry).isNotNull();
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(0);
        Assertions.assertThat((Collection)entry.getAttribute("description")).isNull();
        reader.close();
    }

    @Test(expectedExceptions={NullPointerException.class})
    public void testSetIncludeAttributeDoesntAllowNull() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setIncludeAttribute(null);
        reader.close();
    }

    @Test(expectedExceptions={NoSuchElementException.class})
    public void testSetIncludeBranchWithNoMatch() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setIncludeBranch(DN.valueOf((String)"dc=example,dc=org"));
        try {
            reader.readEntry();
        }
        finally {
            reader.close();
        }
    }

    @Test
    public void testSetIncludeBranchWithMatch() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setIncludeBranch(DN.valueOf((String)"dc=example,dc=com"));
        Entry entry = reader.readEntry();
        reader.close();
        Assertions.assertThat((Object)entry).isNotNull();
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(this.nbStandardEntryAttributes);
    }

    @Test(expectedExceptions={NullPointerException.class})
    public void testSetIncludeBranchDoesntAllowNull() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setIncludeBranch(null);
        reader.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(expectedExceptions={NoSuchElementException.class})
    public void testSetIncludeFilterWithNoMatch() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        Filter filter = Filter.equality((String)"objectclass", (Object)"vip");
        Matcher includeFilter = filter.matcher();
        reader.setIncludeFilter(includeFilter);
        Entry entry = null;
        try {
            entry = reader.readEntry();
        }
        finally {
            reader.close();
        }
        Assertions.assertThat((Object)entry).isNull();
    }

    @Test
    public void testSetIncludeFilterWithMatch() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        Filter filter = Filter.equality((String)"objectclass", (Object)"person");
        Matcher includeFilter = filter.matcher();
        reader.setIncludeFilter(includeFilter);
        Entry entry = reader.readEntry();
        reader.close();
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(this.nbStandardEntryAttributes);
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
        Assertions.assertThat((Collection)entry.getAttribute("cn")).isNotNull();
        Assertions.assertThat((Collection)entry.getAttribute("sn")).isNotNull();
    }

    @Test(expectedExceptions={NullPointerException.class})
    public void testSetIncludeFilterDoesntAllowNull() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setIncludeFilter(null);
        reader.close();
    }

    @Test
    public void testRejectedLDIFListenerMalformedFirstRecord() throws Exception {
        RejectedLDIFListener listener = (RejectedLDIFListener)Mockito.mock(RejectedLDIFListener.class);
        LDIFEntryReader reader = new LDIFEntryReader(new String[]{"dn: baddn", "changetype: add", "objectClass: top", "objectClass: domainComponent", "dc: example"});
        reader.setRejectedLDIFListener(listener);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
        ((RejectedLDIFListener)Mockito.verify((Object)listener)).handleMalformedRecord(Matchers.eq((long)1L), (List)Matchers.eq(Arrays.asList("dn: baddn", "changetype: add", "objectClass: top", "objectClass: domainComponent", "dc: example")), (LocalizableMessage)Matchers.any(LocalizableMessage.class));
        reader.close();
    }

    @Test
    public void testRejectedLDIFListenerMalformedSecondRecord() throws Exception {
        RejectedLDIFListener listener = (RejectedLDIFListener)Mockito.mock(RejectedLDIFListener.class);
        LDIFEntryReader reader = new LDIFEntryReader(new String[]{"dn: dc=example,dc=com", "changetype: add", "objectClass: top", "objectClass: domainComponent", "dc: example", "", "dn: baddn", "changetype: add", "objectClass: top", "objectClass: domainComponent", "dc: example"});
        reader.setRejectedLDIFListener(listener);
        reader.readEntry();
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
        ((RejectedLDIFListener)Mockito.verify((Object)listener)).handleMalformedRecord(Matchers.eq((long)7L), (List)Matchers.eq(Arrays.asList("dn: baddn", "changetype: add", "objectClass: top", "objectClass: domainComponent", "dc: example")), (LocalizableMessage)Matchers.any(LocalizableMessage.class));
        reader.close();
    }

    @Test
    public void testRejectedRecordListenerRejectsBadSchemaRecord() throws Exception {
        RejectedLDIFListener listener = (RejectedLDIFListener)Mockito.mock(RejectedLDIFListener.class);
        LDIFEntryReader reader = new LDIFEntryReader(new String[]{"dn: dc=example,dc=com", "changetype: add", "objectClass: top", "objectClass: domainComponent", "dc: example", "xxx: unknown attribute"});
        reader.setRejectedLDIFListener(listener).setSchemaValidationPolicy(SchemaValidationPolicy.ignoreAll().checkAttributesAndObjectClasses(SchemaValidationPolicy.Action.REJECT));
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
        ((RejectedLDIFListener)Mockito.verify((Object)listener)).handleSchemaValidationFailure(Matchers.eq((long)1L), (List)Matchers.eq(Arrays.asList("dn: dc=example,dc=com", "changetype: add", "objectClass: top", "objectClass: domainComponent", "dc: example", "xxx: unknown attribute")), Matchers.anyListOf(LocalizableMessage.class));
        reader.close();
    }

    @Test
    public void testRejectedLDIFListenerWarnsBadSchemaRecord() throws Exception {
        RejectedLDIFListener listener = (RejectedLDIFListener)Mockito.mock(RejectedLDIFListener.class);
        LDIFEntryReader reader = new LDIFEntryReader(new String[]{"dn: dc=example,dc=com", "changetype: add", "objectClass: top", "objectClass: domainComponent", "dc: example", "xxx: unknown attribute"});
        reader.setRejectedLDIFListener(listener).setSchemaValidationPolicy(SchemaValidationPolicy.ignoreAll().checkAttributesAndObjectClasses(SchemaValidationPolicy.Action.WARN));
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        Entry entry = reader.readEntry();
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"dc=example,dc=com");
        Assertions.assertThat((boolean)entry.containsAttribute("objectClass", new Object[]{"top", "domainComponent"})).isTrue();
        Assertions.assertThat((boolean)entry.containsAttribute("dc", new Object[]{"example"})).isTrue();
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(2);
        ((RejectedLDIFListener)Mockito.verify((Object)listener)).handleSchemaValidationWarning(Matchers.eq((long)1L), (List)Matchers.eq(Arrays.asList("dn: dc=example,dc=com", "changetype: add", "objectClass: top", "objectClass: domainComponent", "dc: example", "xxx: unknown attribute")), Matchers.anyListOf(LocalizableMessage.class));
        reader.close();
    }

    @Test
    public void testRejectedLDIFListenerSkipsRecord() throws Exception {
        RejectedLDIFListener listener = (RejectedLDIFListener)Mockito.mock(RejectedLDIFListener.class);
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setRejectedLDIFListener(listener).setExcludeBranch(DN.valueOf((String)"dc=com"));
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
        ((RejectedLDIFListener)Mockito.verify((Object)listener)).handleSkippedRecord(Matchers.eq((long)1L), (List)Matchers.eq(Arrays.asList(this.getStandardEntry())), (LocalizableMessage)Matchers.any(LocalizableMessage.class));
        reader.close();
    }

    @Test
    public void testSetRejectedLDIFListenerDoesAllowNull() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setRejectedLDIFListener(null);
        Entry entry = reader.readEntry();
        reader.close();
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(this.nbStandardEntryAttributes);
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
    }

    @Test(expectedExceptions={DecodeException.class})
    public void testSetSchemaValidationPolicyDefaultRejectsEntry() throws Exception {
        String[] strEntry = new String[]{"dn: uid=user.0,ou=People,dc=example,dc=com", "objectClass: person", "objectClass: top", "cn: Aaccf Amar", "sn: Amar", "uid: user.0"};
        LDIFEntryReader reader = new LDIFEntryReader(strEntry);
        reader.setSchema(Schema.getDefaultSchema());
        reader.setSchemaValidationPolicy(SchemaValidationPolicy.defaultPolicy());
        try {
            reader.readEntry();
        }
        finally {
            reader.close();
        }
    }

    @Test
    public void testSetSchemaValidationPolicyDefaultAllowsEntry() throws Exception {
        String[] strEntry = new String[]{"dn: uid=user.0,ou=People,dc=example,dc=com", "objectClass: person", "objectClass: top", "cn: Aaccf Amar", "sn: Amar"};
        LDIFEntryReader reader = new LDIFEntryReader(strEntry);
        reader.setSchema(Schema.getDefaultSchema());
        reader.setSchemaValidationPolicy(SchemaValidationPolicy.defaultPolicy());
        Entry entry = reader.readEntry();
        reader.close();
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(3);
    }

    @Test(expectedExceptions={NullPointerException.class})
    public void testSetSchemaValidationPolicyDoesntAllowNull() throws Exception {
        String[] strEntry = new String[]{"dn: uid=user.0,ou=People,dc=example,dc=com", "objectClass: person", "objectClass: top", "cn: Aaccf Amar", "sn: Amar", "uid: user.0"};
        LDIFEntryReader reader = new LDIFEntryReader(strEntry);
        reader.setSchema(Schema.getDefaultSchema());
        reader.setSchemaValidationPolicy(null);
        reader.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSetSchemaSetSchemaValidationPolicyDefaultAllowsEntryWithNewAttribute() throws Exception {
        String[] strEntry = new String[]{"dn: uid=user.0,ou=People,dc=example,dc=com", "objectClass: person", "objectClass: organizationalperson", "objectClass: top", "cn: Aaccf Amar", "sn: Amar", "objectClass: myCustomObjClass", "myCustomAttribute: Testing..."};
        SchemaBuilder scBuild = new SchemaBuilder();
        scBuild.addObjectClass("( temporary-fake-oc-id NAME 'myCustomObjClass' SUP top AUXILIARY MAY myCustomAttribute )", false);
        scBuild.addAttributeType("( temporary-fake-attr-id NAME 'myCustomAttribute' EQUALITY caseIgnoreMatch ORDERING caseIgnoreOrderingMatch SUBSTR caseIgnoreSubstringsMatch SYNTAX 1.3.6.1.4.1.1466.115.121.1.15 USAGE userApplications )", false);
        scBuild.addSchema(Schema.getCoreSchema(), false);
        Schema schema = scBuild.toSchema();
        LDIFEntryReader reader = new LDIFEntryReader(strEntry);
        reader.setSchema(schema);
        reader.setSchemaValidationPolicy(SchemaValidationPolicy.defaultPolicy());
        Entry entry = null;
        try {
            entry = reader.readEntry();
            Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(4);
            Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
            Assertions.assertThat((String)entry.getAttribute("sn").firstValue().toString()).isEqualTo((Object)"Amar");
            Assertions.assertThat((String)entry.getAttribute("cn").firstValueAsString()).isEqualTo((Object)"Aaccf Amar");
            Assertions.assertThat((String)entry.getAttribute(AttributeDescription.valueOf((String)"myCustomAttribute", (Schema)schema)).firstValueAsString()).isEqualTo((Object)"Testing...");
        }
        finally {
            reader.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(expectedExceptions={DecodeException.class})
    public void testSetSchemaSetSchemaValidationPolicyDefaultDoesntAllowEntryWithNewAttribute() throws Exception {
        String[] strEntry = new String[]{"dn: uid=user.0,ou=People,dc=example,dc=com", "objectClass: person", "objectClass: organizationalperson", "objectClass: top", "cn: Aaccf Amar", "sn: Amar", "objectClass: myCustomObjClass", "myCustomAttribute: Testing..."};
        LDIFEntryReader reader = new LDIFEntryReader(strEntry);
        SchemaBuilder scBuild = new SchemaBuilder();
        scBuild.addSchema(Schema.getCoreSchema(), false);
        reader.setSchema(scBuild.toSchema());
        reader.setSchemaValidationPolicy(SchemaValidationPolicy.defaultPolicy());
        try {
            reader.readEntry();
        }
        finally {
            reader.close();
        }
    }

    @Test(expectedExceptions={NullPointerException.class})
    public void testSetSchemaDoesntAllowNull() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader(this.getStandardEntry());
        reader.setSchema(null);
        reader.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(expectedExceptions={NoSuchElementException.class})
    public void testEmpty() throws Exception {
        String path = TestCaseUtils.createTempFile("");
        FileInputStream in = new FileInputStream(path);
        LDIFEntryReader reader = new LDIFEntryReader((InputStream)in);
        try {
            Assert.assertFalse((boolean)reader.hasNext());
            Assert.assertFalse((boolean)reader.hasNext());
            reader.readEntry();
        }
        finally {
            Assert.assertFalse((boolean)reader.hasNext());
            reader.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testReadEntryWithNoSpaces() throws Exception {
        String[] strEntry = new String[]{"# Entry of SCarter", "dn:uid=scarter,ou=People,dc=example,dc=com", "objectClass:person", "objectClass:inetorgperson", "objectClass:organizationalperson", "objectClass:top", "postalAddress:Aaccf Amar$01251 Chestnut Street$Panama City, DE  50369", "postalCode:50369", "uid:scarter", "description::U2hvcnQgZGVzY3JpcHRpb24gb2YgU2NhcnRlcg=="};
        String path = TestCaseUtils.createTempFile(strEntry);
        FileInputStream in = new FileInputStream(path);
        Entry entry = null;
        try (LDIFEntryReader reader = new LDIFEntryReader((InputStream)in);){
            Assertions.assertThat((boolean)reader.hasNext());
            entry = reader.readEntry();
            Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=scarter,ou=People,dc=example,dc=com");
            Assertions.assertThat((String)entry.getAttribute("uid").firstValueAsString()).isEqualTo((Object)"scarter");
            Assertions.assertThat((String)entry.getAttribute("description").firstValueAsString()).isEqualTo((Object)"Short description of Scarter");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testReadEntryWithAttributesSpacesAtStart() throws Exception {
        String[] strEntry = new String[]{"#   Entry of SCarter", "dn:   uid=scarter,ou=People,dc=example,dc=com", "objectClass:   person", "objectClass:   inetorgperson", "objectClass:   organizationalperson", "objectClass:   top", "postalAddress:   Aaccf Amar$01251 Chestnut Street$Panama City, DE  50369", "postalCode:   50369", "uid:    scarter", "description::    U2hvcnQgZGVzY3JpcHRpb24gb2YgU2NhcnRlcg=="};
        String path = TestCaseUtils.createTempFile(strEntry);
        FileInputStream in = new FileInputStream(path);
        Entry entry = null;
        try (LDIFEntryReader reader = new LDIFEntryReader((InputStream)in);){
            Assertions.assertThat((boolean)reader.hasNext());
            entry = reader.readEntry();
            Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=scarter,ou=People,dc=example,dc=com");
            Assertions.assertThat((String)entry.getAttribute("uid").firstValueAsString()).isEqualTo((Object)"scarter");
            Assertions.assertThat((String)entry.getAttribute("description").firstValueAsString()).isEqualTo((Object)"Short description of Scarter");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(expectedExceptions={DecodeException.class})
    public void testReadEntryWithAttributesSpacesAtEnd() throws Exception {
        String[] strEntry = new String[]{"#   Entry of SCarter   ", "dn:   uid=scarter,ou=People,dc=example,dc=com    ", "objectClass:   person    ", "objectClass:   inetorgperson    ", "objectClass:   organizationalperson    ", "objectClass:   top  ", "postalAddress:   Aaccf Amar$01251 Chestnut Street$Panama City, DE  50369  ", "postalCode:   50369 ", "uid:    scarter  ", "description::    U2hvcnQgZGVzY3JpcHRpb24gb2YgU2NhcnRlcg==   "};
        String path = TestCaseUtils.createTempFile(strEntry);
        FileInputStream in = new FileInputStream(path);
        Entry entry = null;
        try (LDIFEntryReader reader = new LDIFEntryReader((InputStream)in);){
            Assertions.assertThat((boolean)reader.hasNext());
            entry = reader.readEntry();
            Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=scarter,ou=People,dc=example,dc=com");
            Assertions.assertThat((String)entry.getAttribute("uid").firstValueAsString()).isEqualTo((Object)"scarter");
            Assertions.assertThat((String)entry.getAttribute("description").firstValueAsString()).isEqualTo((Object)"Short description of Scarter");
        }
    }

    @Test(expectedExceptions={NoSuchElementException.class})
    public void testReadEntry() throws Exception {
        String path = TestCaseUtils.createTempFile(this.getStandardEntry());
        FileInputStream in = new FileInputStream(path);
        try (LDIFEntryReader reader = new LDIFEntryReader((InputStream)in);){
            Assert.assertTrue((boolean)reader.hasNext());
            Entry entry = reader.readEntry();
            Assert.assertNotNull((Object)entry);
            Assert.assertEquals((Iterable)entry.getName(), (Iterable)DN.valueOf((String)"uid=user.0,ou=People,dc=example,dc=com"));
            Assert.assertFalse((boolean)reader.hasNext());
            reader.readEntry();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(expectedExceptions={DecodeException.class})
    public void testLDIFEntryReaderEntryWithDuplicateAttributes() throws Exception {
        String[] strEntry = new String[]{"dn: cn=user.0,ou=People,dc=example,dc=com", "objectClass: organizationalperson", "objectClass: top", "postalAddress: Aaccf Amar$01251 Chestnut Street$Panama City, DE  50369", "postalCode: 50369", "description: This is the description for Aaccf Amar.", "userPassword: ;", "telephoneNumber: +1 685 622 6202", "sn: Amar", "entryUUID: ad55a34a-763f-358f-93f9-da86f9ecd9e4", "entryUUID: ad55a34a-763f-358f-93f9-da45f9ecd9e4", "objectClass: person", "objectClass: person"};
        String path = TestCaseUtils.createTempFile(strEntry);
        FileInputStream in = new FileInputStream(path);
        LDIFEntryReader reader = new LDIFEntryReader((InputStream)in);
        reader.setSchemaValidationPolicy(SchemaValidationPolicy.defaultPolicy());
        try {
            reader.readEntry();
        }
        finally {
            reader.close();
        }
    }

    @Test
    public void testLDIFEntryReaderFullEntry() throws Exception {
        String[] strEntry = new String[]{"version: 1", "dn: cn=Barbara Jensen, ou=Product Development, dc=airius, dc=com", "objectclass: top", "objectclass: person", "objectclass: organizationalPerson", "cn: Barbara Jensen", "cn: Barbara J Jensen", "cn: Babs Jensen", "sn: Jensen", "uid: bjensen", "telephonenumber: +1 408 555 1212", "description: A big sailing fan.", "", "dn: cn=Bjorn Jensen, ou=Accounting, dc=airius, dc=com", "objectclass: top", "objectclass: person", "objectclass: organizationalPerson", "cn: Bjorn Jensen", "sn: Jensen", "telephonenumber: +1 408 555 1212"};
        String path = TestCaseUtils.createTempFile(strEntry);
        FileInputStream in = new FileInputStream(path);
        try (LDIFEntryReader reader = new LDIFEntryReader((InputStream)in);){
            Assertions.assertThat((boolean)reader.hasNext());
            Entry entry = reader.readEntry();
            Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"cn=Barbara Jensen,ou=Product Development,dc=airius,dc=com");
            Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(6);
            entry = reader.readEntry();
            Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"cn=Bjorn Jensen,ou=Accounting,dc=airius,dc=com");
            Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(4);
            Assertions.assertThat((boolean)reader.hasNext()).isFalse();
        }
    }

    @Test
    public void testLDIFEntryReaderMultiplesAttributeValuesDifferentLetterCase() throws Exception {
        String[] strEntry = new String[]{"dn: cn=Character Set,cn=Password Validators,cn=config", "objectClass: ds-cfg-character-set-password-validator", "objectClass: ds-cfg-password-validator", "objectClass: top", "ds-cfg-enabled: true", "ds-cfg-java-class: org.opends.server.extensions.CharacterSetPasswordValidator", "ds-cfg-allow-unclassified-characters: true", "ds-cfg-character-set: 1:abcdefghijklmnopqrstuvwxyz", "ds-cfg-character-set: 1:ABCDEFGHIJKLMNOPQRSTUVWXYZ", "ds-cfg-character-set: 1:0123456789", "ds-cfg-character-set: 1:~!@#$%^&*()-_=+[]{}|;:,.<>/?", "cn: Character Set"};
        String path = TestCaseUtils.createTempFile(strEntry);
        FileInputStream in = new FileInputStream(path);
        try (LDIFEntryReader reader = new LDIFEntryReader((InputStream)in);){
            Assertions.assertThat((boolean)reader.hasNext());
            Entry entry = reader.readEntry();
            Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"cn=Character Set,cn=Password Validators,cn=config");
            Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(6);
            Assertions.assertThat((Collection)entry.getAttribute("ds-cfg-character-set")).isNotEmpty();
            Assertions.assertThat((int)entry.getAttribute("ds-cfg-character-set").toArray().length).isEqualTo(4);
            Assertions.assertThat((boolean)reader.hasNext()).isFalse();
        }
    }

    @Test(expectedExceptions={DecodeException.class})
    public void testValueOfLDIFEntryReadStandardEntryMissingValues() throws Exception {
        String[] strEntry = new String[]{"dn: uid=user.0,ou=People,dc=example,dc=com", "objectClass: person", "objectClass: organizationalperson", "objectClass: top", "cn: Aaccf Amar", "sn:"};
        LDIFEntryReader reader = new LDIFEntryReader(strEntry);
        reader.setSchema(Schema.getDefaultSchema());
        reader.setSchemaValidationPolicy(SchemaValidationPolicy.defaultPolicy());
        try {
            reader.readEntry();
        }
        finally {
            reader.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(expectedExceptions={DecodeException.class})
    public void testValueOfLDIFEntryBERUnexpectedBinaryOption() throws Exception {
        String[] strEntry = new String[]{"version: 1", "dn:: b3U95Za25qWt6YOoLG89QWlyaXVz", "# dn:: ou=<JapaneseOU>,o=Airius", "objectclass: top", "objectclass: person", "objectclass: organizationalPerson", "cn: Horatio Jensen", "cn: Horatio N Jensen", "sn: Jensen", "uid: hjensen", "sn;binary:: 5bCP56yg5Y6f"};
        LDIFEntryReader reader = new LDIFEntryReader(strEntry);
        Schema schema = Schema.getCoreSchema();
        reader.setSchema(schema);
        reader.setSchemaValidationPolicy(SchemaValidationPolicy.defaultPolicy());
        try {
            reader.readEntry();
        }
        finally {
            reader.close();
        }
    }

    @Test(expectedExceptions={DecodeException.class})
    public void testValueOfLDIFEntryFatalContinuationLineAtStart() throws Exception {
        String[] strEntry = new String[]{" This is a fatal continuation line at start", "dn:: b3U95Za25qWt6YOoLG89QWlyaXVz", "# dn:: ou=<JapaneseOU>,o=Airius", "objectclass: top", "objectclass: person", "objectclass: organizationalPerson", "cn: Horatio Jensen", "cn: Horatio N Jensen", "sn: Jensen", "uid: hjensen"};
        try (LDIFEntryReader reader = new LDIFEntryReader(strEntry);){
            reader.readEntry();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testValueOfLDIFEntryReadEntryContainingURL() throws Exception {
        File file = File.createTempFile("sdk", ".jpeg");
        String url = file.toURI().toURL().toString();
        LDIFEntryReader reader = new LDIFEntryReader(new String[]{"#A single comment", " continued in the second line", "version: 1", "dn:: b3U95Za25qWt6YOoLG89QWlyaXVz", "# dn:: ou=<JapaneseOU>,o=Airius", "objectclass: top", "objectclass: person", "objectclass: organizationalPerson", "cn: Horatio Jensen", "cn: Horatio N Jensen", "sn: Jensen", "uid: hjensen", "telephonenumber: +1 408 555 1212", "jpegphoto:< " + url, "#This is a end line comment", "# Followed by another"});
        try {
            Entry entry = reader.readEntry();
            Assertions.assertThat((String)entry.getName().toString()).isNotEqualTo((Object)"b3U95Za25qWt6YOoLG89QWlyaXVz");
            Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(6);
            Assertions.assertThat((Collection)entry.getAttribute("jpegphoto")).isNotEmpty();
            Assertions.assertThat((String)entry.getAttribute("cn").firstValueAsString()).isEqualTo((Object)"Horatio Jensen");
        }
        finally {
            file.delete();
            reader.close();
        }
    }

    @Test(expectedExceptions={DecodeException.class})
    public void testValueOfLDIFEntryReadEntryContainingMalformedURL() throws Exception {
        try (LDIFEntryReader reader = new LDIFEntryReader(new String[]{"version: 1", "dn:: b3U95Za25qWt6YOoLG89QWlyaXVz", "# dn:: ou=<JapaneseOU>,o=Airius", "objectclass: top", "objectclass: person", "objectclass: organizationalPerson", "cn: Horatio Jensen", "cn: Horatio N Jensen", "sn: Jensen", "uid: hjensen", "telephonenumber: +1 408 555 1212", "jpegphoto:< invalidProtocol", " ", " "});){
            reader.readEntry();
        }
    }

    @Test(expectedExceptions={DecodeException.class})
    public void testReadEntryParseColonPositionThrowException() throws Exception {
        String path = TestCaseUtils.createTempFile("#Entry made for testing", ": cn=Gern Jensen, ou=Product Testing, dc=airius, dc=com", "objectclass: top", "objectclass: person", "objectclass: organizationalPerson");
        FileInputStream in = new FileInputStream(path);
        try (LDIFEntryReader reader = new LDIFEntryReader((InputStream)in);){
            reader.readEntry();
        }
    }

    @Test(expectedExceptions={DecodeException.class})
    public void testReadEntryBase64EncodedMalformedBase64Attribute() throws Exception {
        try (LDIFEntryReader reader = new LDIFEntryReader(Arrays.asList("version: 1", "dn: cn=Gern Jensen, ou=Product Testing, dc=airius, dc=com", "objectclass: top", "objectclass: person", "objectclass: organizationalPerson", "cn: Gern Jensen", "cn: Gern O Jensen", "sn: Jensen", "uid: gernj", "telephonenumber: +1 408 555 1212", "description:: V2hhdCBhIGNhcmVmdWwgcmVhZGVyIHlvdSBhcmUhICBUaGlzIHZhbHVlIGlzIGJhc2UtNjQtZW5aaaaaaaaaaajb2RlZCBiZWNhdXNlIGl0IGhhcyBhIGNvbnRyb2wgY2hhcmFjdGVyIGluIGl0IChhIENSKS4NICBCeSB0aGUgd2F5LCB5b3Ugc2hvdWxkIHJlYWxseSBnZXQgb3V0IG1vcmUu"));){
            reader.readEntry();
        }
    }

    @Test
    public void testReadEntryBase64Encoded() throws Exception {
        try (LDIFEntryReader reader = new LDIFEntryReader(Arrays.asList("version: 1", "dn: cn=Gern Jensen, ou=Product Testing, dc=airius, dc=com", "objectclass: top", "objectclass: person", "objectclass: organizationalPerson", "cn: Gern Jensen", "cn: Gern O Jensen", "sn: Jensen", "uid: gernj", "telephonenumber: +1 408 555 1212", "description:: V2hhdCBhIGNhcmVmdWwgcmVhZGVyIHlvdSBhcmUhICBUaGlzIHZhbHVlIGlzIGJhc2UtNjQtZW5jb2RlZCBiZWNhdXNlIGl0IGhhcyBhIGNvbnRyb2wgY2hhcmFjdGVyIGluIGl0IChhIENSKS4NICBCeSB0aGUgd2F5LCB5b3Ugc2hvdWxkIHJlYWxseSBnZXQgb3V0IG1vcmUu"));){
            Assertions.assertThat((boolean)reader.hasNext());
            Entry entry = reader.readEntry();
            Assertions.assertThat((Object)entry).isNotNull();
            Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(6);
            Assertions.assertThat((String)entry.getAttribute("cn").firstValueAsString()).isEqualTo((Object)"Gern Jensen");
            Assertions.assertThat((String)entry.getAttribute("description").firstValueAsString()).isNotSameAs((Object)"V2hhdCBhIGNhcmVmdWwgcmVhZGVyIHlvdSBhcmUhICBUaGlzIHZhbHVlIGlzIGJhc2UtNjQtZW5jb2RlZCBiZWNhdXNlIGl0IGhhcyBhIGNvbnRyb2wgY2hhcmFjdGVyIGluIGl0IChhIENSKS4NICBCeSB0aGUgd2F5LCB5b3Ugc2hvdWxkIHJlYWxseSBnZXQgb3V0IG1vcmUu");
            Assertions.assertThat((String)entry.getAttribute("description").firstValueAsString()).contains("What a careful reader you are!");
        }
    }

    @Test
    public void testReadEntryBase64EncodedDN() throws Exception {
        try (LDIFEntryReader reader = new LDIFEntryReader(Arrays.asList("dn::  dWlkPXJvZ2FzYXdhcmEsb3U95Za25qWt6YOoLG89QWlyaXVz", "# dn:: uid=<uid>,ou=<JapaneseOU>,o=Airius", "objectclass: top", "objectclass: person", "objectclass: organizationalPerson", "cn: Gern Jensen", "cn: Gern O Jensen", "sn: Jensen", "uid: gernj"));){
            Assertions.assertThat((boolean)reader.hasNext());
            Entry entry = reader.readEntry();
            Assertions.assertThat((boolean)reader.hasNext()).isFalse();
            Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=rogasawara,ou=\u55b6\u696d\u90e8,o=Airius");
        }
    }

    @Test(expectedExceptions={DecodeException.class})
    public void testReadEntryBase64EncodedDNMalformedThrowsError() throws Exception {
        try (LDIFEntryReader reader = new LDIFEntryReader(Arrays.asList("dn:: dWlkPXJvZ2FzYXdh!!!OOOpppps!!!25qWt6YOoLG89QWlyaXVz", "# dn:: uid=<uid>,ou=<JapaneseOU>,o=Airius", "objectclass: top", "objectclass: person", "objectclass: organizationalPerson", "cn: Gern Jensen", "cn: Gern O Jensen", "sn: Jensen", "uid: gernj"));){
            reader.readEntry();
        }
    }

    @Test
    public void testLDIFEntryReaderEntryAsArray() throws Exception {
        try (LDIFEntryReader reader = new LDIFEntryReader(Arrays.asList(this.getStandardEntry()));){
            Assertions.assertThat((boolean)reader.hasNext());
            Assertions.assertThat((int)reader.readEntry().getAttributeCount()).isEqualTo(this.nbStandardEntryAttributes);
        }
    }

    @Test(expectedExceptions={NullPointerException.class})
    public void testLDIFEntryReaderInpuStreamDoesntAllowNull() throws Exception {
        LDIFEntryReader reader = new LDIFEntryReader((InputStream)null);
        reader.close();
    }

    @Test(expectedExceptions={IOException.class})
    public void testReadEntryThrowsIOException() throws Exception {
        FileInputStream mockIn = (FileInputStream)Mockito.mock(FileInputStream.class);
        LDIFEntryReader reader = new LDIFEntryReader((InputStream)mockIn);
        ((FileInputStream)Mockito.doThrow((Throwable)new IOException()).when((Object)mockIn)).read();
        try {
            reader.readEntry();
        }
        finally {
            reader.close();
            ((FileInputStream)Mockito.verify((Object)mockIn, (VerificationMode)Mockito.times((int)1))).close();
        }
    }

    @Test(expectedExceptions={LocalizedIllegalArgumentException.class})
    public void testValueOfLDIFEntryMultipleChangeRecordFound() throws Exception {
        LDIFEntryReader.valueOfLDIFEntry((String[])new String[]{"#This is an example test", "dn: CN=John Smith,OU=Legal,DC=example,DC=com", "changetype: modify", "replace:employeeID", "employeeID: 1234", "", "dn: CN=Jane Smith,OU=Accounting,DC=example,DC=com", "changetype: modify", "replace:employeeID", "employeeID: 5678"});
    }

    @Test(expectedExceptions={LocalizedIllegalArgumentException.class})
    public void testValueOfLDIFEntryThrowsExceptionIfOnlyAComment() throws Exception {
        LDIFEntryReader.valueOfLDIFEntry((String[])new String[]{"#This is an example test"});
    }

    @Test(expectedExceptions={LocalizedIllegalArgumentException.class})
    public void testValueOfLDIFEntryMalformedEntry() throws Exception {
        LDIFEntryReader.valueOfLDIFEntry((String[])new String[]{"objectClass: top", "objectClass: person", "objectClass: organizationalperson", "objectClass: inetorgperson"});
    }

    @Test
    public void testValueOfLDIFEntryWellFormedEntry() throws Exception {
        Entry entry = LDIFEntryReader.valueOfLDIFEntry((String[])new String[]{"dn: uid=user.0,ou=People,dc=example,dc=com", "objectClass: top", "objectClass: person", "objectClass: organizationalperson", "objectClass: inetorgperson"});
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(1);
    }

    @Test
    public void testValueOfLDIFEntryReadStandardEntry() throws Exception {
        Entry entry = LDIFEntryReader.valueOfLDIFEntry((String[])this.getStandardEntry());
        Assertions.assertThat((Object)entry).isNotNull();
        Assertions.assertThat((Iterable)entry.getName()).isNotNull();
        Assertions.assertThat((String)entry.getName().toString()).isEqualTo((Object)"uid=user.0,ou=People,dc=example,dc=com");
        Assertions.assertThat((String)entry.getAttribute("sn").firstValue().toString()).isEqualTo((Object)"Amar");
        Assertions.assertThat((int)entry.getAttributeCount()).isEqualTo(this.nbStandardEntryAttributes);
    }

    @Test(expectedExceptions={NullPointerException.class})
    public void testValueOfLDIFEntryDoesntAllowNull() throws Exception {
        LDIFEntryReader.valueOfLDIFEntry((String[])null);
    }
}

